<?php
/*------------------------------------------------------------------------------
  $Id$

  AbanteCart, Ideal OpenSource Ecommerce Solution
  http://www.AbanteCart.com

  Copyright © 2011-2013 Belavier Commerce LLC

  This source file is subject to Open Software License (OSL 3.0)
  License details is bundled with this package in the file LICENSE.txt.
  It is also available at this URL:
  &lt;http://www.opensource.org/licenses/OSL-3.0&gt;
  
 UPGRADE NOTE: 
   Do not edit or add to this file if you wish to upgrade AbanteCart to newer
   versions in the future. If you wish to customize AbanteCart for your
   needs please refer to http://www.AbanteCart.com for more information.  
------------------------------------------------------------------------------*/

if (!defined('DIR_CORE')) {
    header('Location: static_pages/');
}

class ModelProductRefinedSearch extends Model
{
    public $data = [];
    //filter object for current instance
    public $filter_params, $filter;

    public function __construct($registry)
    {
        parent::__construct($registry);

        //load filter
        $this->filter_params = [
            'category_id',
            'manufacturer_id',
            'keyword',
            'tag',
            'specials',
            'free_shipping',
            'match',
            'pfrom',
            'pto',
            'wfrom',
            'wto',
        ];
        $grid_filter_params = ['name', 'description', 'model', 'sku'];
        $this->data['filter_data'] = [
            'method'             => 'get',
            'filter_params'      => $this->filter_params,
            'grid_filter_params' => $grid_filter_params,
        ];

        if (isset($this->registry->get('request')->get['path'])) {
            $path = explode('_', $this->registry->get('request')->get['path']);
            $this->registry->get('request')->get['category_id'] = end($path);
        }
        $this->extensions->hk_InitData($this, __FUNCTION__);
        $this->filter = new AFilter($this->data['filter_data']);
    }

    /**
     * @param array $data
     * @param string $mode
     *
     * @return false|int|array|null
     * @throws AException
     */
    public function getProducts($data = [], $mode = 'default')
    {
        //make sure we have data to filter. Try to load filter object
        $data = array_merge($this->filter->getFilterData(), (array) $data);

        //still no data? return nothing
        if (!$data) {
            return null;
        }

        if (empty($data['content_language_id'])) {
            $data['content_language_id'] = (int) $this->config->get('storefront_language_id');
        }
        $data['store_id'] = (int) $this->config->get('config_store_id');
        //check cache 
        $cacheKey = 'product.refine.list.'.$this->buildСacheKey($data, $mode);
        $cache_data = $this->cache->pull($cacheKey);
        if ($cache_data !== false) {
            return $cache_data;
        }

        if ($mode == 'total_only') {
            $sql = "SELECT COUNT(DISTINCT p.product_id) as total ";
        } else {
            $sql = "SELECT DISTINCT pd.*, p.*, ".$this->_sql_final_price_string().", ".$this->_sql_avg_rating_string();
        }

        $sql .= $this->_build_common_joins($data);
        $sql .= $this->_build_sql($data);
        //group by to get unique set
        //If for total, we done building the query
        if ($mode == 'total_only') {
            $query = $this->db->query($sql);
            $this->cache->push($cacheKey, $query->row['total']);
            return $query->row['total'];
        }

        $sort_data = [
            'name'       => 'pd.name',
            'model'      => 'p.model',
            'sku'      => 'p.sku',
            'quantity'   => 'p.quantity',
            'price'      => 'final_price',
            'p.price'    => 'final_price',
            'special'    => 'final_price',
            'rating'     => 'rating',
            'status'     => 'p.status',
            'sort_order' => 'p.sort_order',
        ];

        if (isset($data['sort']) && in_array($data['sort'], array_keys($sort_data))) {
            $sql .= " ORDER BY ".$sort_data[$data['sort']];
        } else {
            $sql .= " ORDER BY pd.name";
        }

        if (isset($data['order']) && ($data['order'] == 'DESC')) {
            $sql .= " DESC";
        } else {
            $sql .= " ASC";
        }

        if (isset($data['start']) || isset($data['limit'])) {
            if ($data['start'] < 0) {
                $data['start'] = 0;
            }

            if ($data['limit'] < 1) {
                $data['limit'] = 20;
            }

            $sql .= " LIMIT ".(int) $data['start'].",".(int) $data['limit'];
        }

        $query = $this->db->query($sql);
        $product_data = $query->rows;
        $this->cache->push($cacheKey, $product_data);
        return $product_data;
    }

    /**
     * @param array $data
     *
     * @return int
     * @throws AException
     */
    public function getTotalProducts($data = [])
    {
        return $this->getProducts($data, 'total_only');
    }

    /**
     * @param array $data
     * @param string $mode
     *
     * @return false|int|array|null
     * @throws AException
     */
    public function getCategories($data = [], $mode = '')
    {
        //make sure we have data to filter. Try to load filter object
        if (!$data) {
            $data = $this->filter->getFilterData();
        }
        //still no data? return nothing
        if (!$data) {
            return null;
        }

        if (!empty($data['content_language_id'])) {
            $language_id = ( int ) $data['content_language_id'];
        } else {
            $data['content_language_id'] = $language_id = (int) $this->config->get('storefront_language_id');
        }
        $data['store_id'] = (int) $this->config->get('config_store_id');
        //check cache 
        $cacheKey = 'category.refine.categories.'.$this->buildСacheKey($data, $mode);
        $cache_data = $this->cache->pull($cacheKey);
        if ($cache_data !== false) {
            return $cache_data;
        }

        if ($mode == 'total_only') {
            $sql = "SELECT COUNT(*) as total ";
        } else {
            $sql = "SELECT c.category_id, cd.name, count(DISTINCT p.product_id) as total_products ";
        }

        $sql .= $this->_build_common_joins($data);

        $sql .= " INNER JOIN ".$this->db->table("categories")." c 
                    ON (c.category_id = p2c.category_id)";
        $sql .= " INNER JOIN ".$this->db->table("category_descriptions")." cd 
                    ON (c.category_id = cd.category_id AND cd.language_id = ".$language_id.")";

        $sql .= $this->_build_sql($data);

        $sql .= " AND c.status=1 
                GROUP BY c.category_id ";

        $query = $this->db->query($sql);
        $categories_data = $query->rows;

        $this->cache->push($cacheKey, $categories_data);

        return $categories_data;
    }

    /**
     * @param array $data
     * @param string $mode
     *
     * @return false|int|array|null
     * @throws AException
     */
    public function getManufacturers($data = [], $mode = '')
    {
        //make sure we have data to filter. Try to load filter object
        $data = array_merge($this->filter->getFilterData(), (array) $data);

        //still no data? return nothing
        if (!$data) {
            return null;
        }

        if (empty($data['content_language_id'])) {
            $data['content_language_id'] = (int) $this->config->get('storefront_language_id');
        }
        $data['store_id'] = $store_id = (int) $this->config->get('config_store_id');
        //check cache 
        $cacheKey = 'manufacturer.refine.brands.'.$this->buildСacheKey($data, $mode);
        $cache_data = $this->cache->pull($cacheKey);
        if ($cache_data !== false) {
            return $cache_data;
        }

        if ($mode == 'total_only') {
            $sql = "SELECT COUNT(*) as total ";
        } else {
            $sql = "SELECT m.manufacturer_id, m.name, count(DISTINCT p.product_id) as total_products ";
        }

        $sql .= $this->_build_common_joins($data);

        $sql .= " INNER JOIN ".$this->db->table("manufacturers")." m
                        ON (p.manufacturer_id = m.manufacturer_id )";
        $sql .= " INNER JOIN ".$this->db->table("manufacturers_to_stores")." m2s
                        ON (m.manufacturer_id = m2s.manufacturer_id AND m2s.store_id = ".$store_id.")";

        $sql .= $this->_build_sql($data);

        $sql .= " GROUP BY m.manufacturer_id";
        if ($mode != 'total_only') {
            $sql .= " ORDER BY m.name ";
        }

        $query = $this->db->query($sql);
        $categories_data = $query->rows;
        $this->cache->push($cacheKey, $categories_data);
        return $categories_data;
    }

    /**
     * @param array $data
     * @param string $mode
     *
     * @return false|mixed|null
     * @throws AException
     */
    public function getTags($data = [], $mode = '')
    {
        //make sure we have data to filter. Try to load filter object
        $data = array_merge($this->filter->getFilterData(), (array) $data);
        //still no data? return nothing
        if (!$data) {
            return null;
        }

        if (!empty($data['content_language_id'])) {
            $language_id = ( int ) $data['content_language_id'];
        } else {
            $data['content_language_id'] = $language_id = (int) $this->config->get('storefront_language_id');
        }
        $data['store_id'] = (int) $this->config->get('config_store_id');
        //check cache 
        $cacheKey = 'product.refine.tag.'.$this->buildСacheKey($data, $mode);
        $cache_data = $this->cache->pull($cacheKey);
        if ($cache_data !== false) {
            return $cache_data;
        }

        if ($mode == 'total_only') {
            $sql = "SELECT COUNT(*) as total ";
        } else {
            $sql = "SELECT pt.tag as name, count(DISTINCT p.product_id) as total_products ";
        }

        $sql .= $this->_build_common_joins($data);
        $sql .= " INNER JOIN ".$this->db->table("product_tags")." pt 
                    ON (p.product_id = pt.product_id and pt.language_id = '".$language_id."')";
        $sql .= $this->_build_sql($data);
        $sql .= " GROUP BY pt.tag ";

        $query = $this->db->query($sql);
        $categories_data = $query->rows;
        $this->cache->push($cacheKey, $categories_data);
        return $categories_data;
    }

    /**
     * @param array $data
     *
     * @return array
     * @throws AException
     */
    public function getPriceRange($data = [])
    {
        //make sure we have data to filter. Try to load filter object
        $data = array_merge($this->filter->getFilterData(), (array) $data);

        $sql = "SELECT MIN(p.price) as min_price, MAX(p.price) as max_price ";

        $sql .= $this->_build_common_joins($data);
        $sql .= $this->_build_sql($data);

        $query = $this->db->query($sql);
        return [
            'min_price' => $this->currency->convert(
                $query->row['min_price'],
                $this->config->get('config_currency'),
                $this->currency->getCode()
            ),
            'max_price' => $this->currency->convert(
                $query->row['max_price'],
                $this->config->get('config_currency'),
                $this->currency->getCode()
            ),
        ];
    }

    /**
     * @return float
     * @throws AException
     */
    public function getMaxPrice()
    {
        $language_id = $this->language->getLanguageID();
        $store_id = (int) $this->config->get('config_store_id');

        //check cache
        $cache_name = 'product.refine.maxprice.'
            .$this->currency->getCode().'.'
            .$language_id.'.'
            .$store_id;

        $cache_data = $this->cache->pull($cache_name);
        if ($cache_data !== false) {
            return $cache_data;
        }

        $sql = "SELECT MAX(p.price) as max_price ";

        $sql .= $this->_build_common_joins();
        $sql .= $this->_build_sql();

        $query = $this->db->query($sql);
        $price = $this->currency->convert(
            $query->row['max_price'],
            $this->config->get('config_currency'),
            $this->currency->getCode()
        );

        $this->cache->push($cache_name, $price);

        return $price;
    }

    /**
     * @param array $data
     *
     * @return string
     */
    protected function _build_common_joins($data = [])
    {
        if (!empty($data['content_language_id'])) {
            $language_id = ( int ) $data['content_language_id'];
        } else {
            $language_id = (int) $this->config->get('storefront_language_id');
        }
        $sql = "FROM ".$this->db->table("products")." p
                INNER JOIN ".$this->db->table("product_descriptions")." pd
                    ON (p.product_id = pd.product_id AND pd.language_id = ".$language_id." )";
        $sql .= " INNER JOIN ".$this->db->table("products_to_categories")." p2c
                    ON (p.product_id = p2c.product_id)";
        return $sql;
    }

    protected function _build_sql($data = [])
    {
        $sql = '';
        $filter = ($data['filter'] ?? []);
        if (!empty($data['content_language_id'])) {
            $language_id = ( int ) $data['content_language_id'];
        } else {
            $language_id = (int) $this->config->get('storefront_language_id');
        }

        //JOIN section 
        if (isset($filter['tag']) && !is_null($filter['tag'])) {
            $sql .= " INNER JOIN ".$this->db->table("product_tags")
                ." p2t ON (p.product_id = p2t.product_id and p2t.language_id = '".$language_id."')";
        }
        if (isset($filter['specials']) && !is_null($filter['specials'])) {
            $sql .= " INNER JOIN ".$this->db->table("product_specials")." psp ON (p.product_id = psp.product_id )";
        }

        //Add SQL JOIN from extensions

        //WHERE section of SQL
        $sql .= " WHERE p.date_available <= NOW() AND p.status = '1' ";

        if (!empty($data['subsql_filter'])) {
            $sql .= " AND ".$data['subsql_filter'];
        }

        if (isset($filter['match']) && !is_null($filter['match'])) {
            $match = $filter['match'];
        } else {
            $match = 'exact';
        }

        if (isset($filter['keyword']) && !is_null($filter['keyword'])) {
            $keywords = explode(' ', $filter['keyword']);

            if ($match == 'any') {
                $sql .= " AND (";
                foreach ($keywords as $k => $keyword) {
                    $sql .= $k > 0 ? " OR" : "";
                    $sql .= " (LCASE(pd.name) LIKE '%".$this->db->escape(strtolower($keyword))."%'";
                    $sql .= " OR LCASE(p.model) LIKE '%".$this->db->escape(strtolower($keyword))."%'";
                    $sql .= " OR LCASE(p.sku) LIKE '%".$this->db->escape(strtolower($keyword))."%')";
                }
                $sql .= " )";
            } else {
                if ($match == 'all') {
                    $sql .= " AND (";
                    foreach ($keywords as $k => $keyword) {
                        $sql .= $k > 0 ? " AND" : "";
                        $sql .= " (LCASE(pd.name) LIKE '%".$this->db->escape(strtolower($keyword))."%'";
                        $sql .= " OR LCASE(p.model) LIKE '%".$this->db->escape(strtolower($keyword))."%'";
                        $sql .= " OR LCASE(p.sku) LIKE '%".$this->db->escape(strtolower($keyword))."%')";
                    }
                    $sql .= " )";
                } else {
                    if ($match == 'exact') {
                        $sql .= " AND (LCASE(pd.name) LIKE '%".$this->db->escape(strtolower($filter['keyword']))."%'";
                        $sql .= " OR LCASE(p.model) LIKE '%".$this->db->escape(strtolower($filter['keyword']))."%'";
                        $sql .= " OR LCASE(p.sku) LIKE '%".$this->db->escape(strtolower($filter['keyword']))."%')";
                    }
                }
            }
        }

        //price filter
        if (isset($filter['pfrom']) && !is_null($filter['pfrom'])) {
            $fname = $this->_sql_final_price_string();
            $fname = str_replace(' as final_price', '', $fname);
            $sql .= " AND ".$fname." >= '".$this->currency->convert(
                    (float) $filter['pfrom'],
                    $this->currency->getCode(),
                    $this->config->get('config_currency')
                )."'";
        }
        if (isset($filter['pto']) && !is_null($filter['pto'])) {
            $fname = $this->_sql_final_price_string();
            $fname = str_replace(' as final_price', '', $fname);

            $sql .= " AND ".$fname." <= '".$this->currency->convert(
                    (float) $filter['pto'],
                    $this->currency->getCode(),
                    $this->config->get('config_currency')
                )."'";
        }

        //weight filter
        if (isset($filter['wfrom']) && !is_null($filter['wfrom'])) {
            $sql .= " AND p.weight >= '".(float) $filter['wfrom']."'";
        }
        if (isset($filter['wto']) && !is_null($filter['wto'])) {
            $sql .= " AND p.weight <= '".(float) $filter['wto']."'";
        }

        //filter by free shipping
        if (isset($filter['free_shipping']) && !is_null($filter['free_shipping'])) {
            $sql .= " AND p.free_shipping = '".(int) $filter['free_shipping']."'";
        }

        if (is_array($filter['category_id']) && count($filter['category_id'])) {
            $sql .= " AND p2c.category_id in (".join(",", $filter['category_id']).")";
        } else {
            if (isset($filter['category_id']) && !is_null($filter['category_id'])) {
                $sql .= " AND p2c.category_id = '".(int) $filter['category_id']."'";
            }
        }

        if (is_array($filter['manufacturer_id']) && count($filter['manufacturer_id'])) {
            $sql .= " AND p2c.manufacturer_id in (".join(",", $filter['manufacturer_id']).")";
        } else {
            if (isset($filter['manufacturer_id']) && !is_null($filter['manufacturer_id'])) {
                $sql .= " AND p.manufacturer_id = '".(int) $filter['manufacturer_id']."'";
            }
        }

        if (is_array($filter['tag']) && count($filter['tag'])) {
            $sql .= " AND LCASE(p2t.tag) in ('".strtolower(join("','", $filter['tag']))."')";
        } else {
            if (isset($filter['tag']) && !is_null($filter['tag'])) {
                $sql .= " AND LCASE(p2t.tag) = '".$this->db->escape(strtolower($filter['tag']))."'";
            }
        }

        //Add SQL from extensions

        return $sql;
    }

    protected function _sql_avg_rating_string()
    {
        return " ( SELECT AVG(r.rating)
                         FROM ".$this->db->table("reviews")." r
                         WHERE p.product_id = r.product_id
                         GROUP BY r.product_id 
                 ) AS rating ";
    }

    protected function _sql_final_price_string()
    {
        //special prices
        if ($this->customer->isLogged()) {
            $customer_group_id = (int) $this->customer->getCustomerGroupId();
        } else {
            $customer_group_id = (int) $this->config->get('config_customer_group_id');
        }

        $sql = " ( SELECT p2sp.price
                    FROM ".$this->db->table("product_specials")." p2sp
                    WHERE p2sp.product_id = p.product_id
                            AND p2sp.customer_group_id = '".$customer_group_id."'
                            AND ((p2sp.date_start = '0000-00-00' OR p2sp.date_start < NOW())
                            AND (p2sp.date_end = '0000-00-00' OR p2sp.date_end > NOW()))
                    ORDER BY p2sp.priority ASC, p2sp.price ASC LIMIT 1
                 ) ";
        return "COALESCE( ".$sql.", p.price) as final_price";
    }

    //Wrapper for filter method
    public function buildFilterURI($exclude_list = [])
    {
        return $this->filter->buildFilterURI($exclude_list);
    }

    //Build cache name based on sorted params

    /**
     * @param array $data
     * @param string $mode
     *
     * @return string
     */
    protected function buildСacheKey($data, $mode)
    {
        //build cache hash name based on parameters
        $string = $mode.'.'
            .$data['sort'].'.'
            .$data['order'].'.'
            .$data['limit'].'.'
            .$data['start'].'.'
            .$data['content_language_id'].'.'
            .$data['store_id'];
        if ($data['filter']) {
            foreach ($data['filter'] as $key => $value) {
                if (is_array($value)) {
                    asort($value);
                    $string .= '.'.$key.'_'.implode('.', $value);
                } else {
                    $string .= '.'.$key.'_'.$value;
                }
            }
        }
        return md5($string);
    }

}
