<?php
/*------------------------------------------------------------------------------

  For Abante Cart, E-commerce Solution
  http://www.AbanteCart.com

  Copyright (c) 2014-2023 We Hear You 2, Inc.  (WHY2)

------------------------------------------------------------------------------*/

if (!defined('DIR_CORE')) {
    header('Location: static_pages/');
}

/**
 * Class ModelExtensionPirateShipIntegration
 */

class ModelExtensionPirateShipIntegration extends Model {

    /**
     * @param array $data
     * @param string $mode
     *
     * @return int|array
     * @throws AException
     */
    public function getOrders($data = [], $mode = 'default') {
        $language_id = $this->language->getLanguageID();

        if (array_key_exists('store_id', $data)) {
            $store_id = $data['store_id'];
        } else {
            $store_id = (int)$this->getStoreId();
        }

        if ($mode == 'total_only') {
            $total_sql = 'count(*) as total';
        } else {
            $total_sql = "o.order_id,
                        CONCAT(o.firstname, ' ', o.lastname) AS name,
                        o.email,
                        o.shipping_company,
                        o.shipping_address_1,
                        o.shipping_address_2,
                        o.shipping_city,
                        o.shipping_address_1,
                        o.shipping_zone_id,
                        o.shipping_postcode,
                        o.shipping_country_id,
                        o.date_added,
                        o.shipping_method,
                        (SELECT shp.shipped
                         FROM ".$this->db->table("pirateship_integration_order_shipped_status")." shp
                         WHERE shp.order_id = o.order_id) AS shipped,
                         (SELECT shp.tracking_number
                         FROM ".$this->db->table("pirateship_integration_order_shipped_status")." shp
                         WHERE shp.order_id = o.order_id) AS tracking_number,
                        (SELECT os.name
                         FROM ".$this->db->table("order_statuses")." os
                         WHERE os.order_status_id = o.order_status_id
                            AND os.language_id = '".(int) $language_id."') AS status,
                         o.order_status_id";
        }

        $sql = "SELECT ".$total_sql."
                FROM `".$this->db->table("orders")."`o";

        if (($data['filter']['order_status_id'] ?? '') == 'all') {
            $sql .= " WHERE o.order_status_id >= 0";
        } else {
            if (isset($data['filter']['order_status_id'])) {
                $sql .= " WHERE o.order_status_id = '".(int) $data['filter']['order_status_id']."'";
            } else {
                $sql .= " WHERE o.order_status_id > '0'";
            }
        }

        if (isset($data['filter']['order_from'])) {
            $data['order_from'] = date_format(date_create($data['filter']['order_from']),'Y-m-d');
            $sql .= " AND DATE(o.date_added) >=DATE('".$this->db->escape($data['order_from'])."')";
        }

        if (isset($data['filter']['order_to'])) {
            $data['order_to'] = date_format(date_create($data['filter']['order_to']),'Y-m-d');
            $sql .= " AND DATE(o.date_added) <= DATE('".$this->db->escape($data['order_to'])."')";
        }

        if ($store_id !== null) {
            $sql .= " AND o.store_id = '".(int) $store_id."'";
        }

        $sql .= " AND (o.shipping_method LIKE '%USPS%' OR o.shipping_method LIKE '%UPS%' OR o.shipping_method_key LIKE '%usps%' OR o.shipping_method_key LIKE '%ups%')";

        //If for total, we done building the query
        if ($mode == 'total_only') {
            $query = $this->db->query($sql);
            return $query->row['total'];
        }

        $sort_data = [
            'o.order_id',
            'name',
            'status',
            'o.date_added',
            'o.total',
        ];

        if (isset($data['sort']) && in_array($data['sort'], $sort_data)) {
            $sql .= " ORDER BY ".$data['sort'];
        } else {
            $sql .= " ORDER BY o.order_id";
        }

        if (isset($data['order']) && (strtoupper($data['order']) == 'DESC')) {
            $sql .= " DESC";
        } else {
            $sql .= " ASC";
        }

        if (isset($data['start']) || isset($data['limit'])) {
            if ($data['start'] < 0) {
                $data['start'] = 0;
            }

            if ($data['limit'] < 1) {
                $data['limit'] = 20;
            }

            $sql .= " LIMIT ".(int) $data['start'].",".(int) $data['limit'];
        }

        $query = $this->db->query($sql);
        $result_rows = [];

        foreach ($query->rows as $row) {
            $result_rows[] = $this->dcrypt->decrypt_data($row, 'orders');
        }
        return $result_rows;
    }

    /**
     * @param array $data
     *
     * @return int
     * @throws AException
     */
    public function getTotalOrders($data = []) {
        return $this->getOrders($data, 'total_only');
    }

    /**
     * @param $data
     * @return void
     * @throws AException
     */
    public function updateStatus($data) {
        $query = $this->db->query("SELECT `order_id` FROM ".$this->db->table("pirateship_integration_order_shipped_status")." WHERE `order_id`=".(int)$data['order_id']);
        $result = $query->row;
        if ($result) {
            $this->db->query("UPDATE " . $this->db->table("pirateship_integration_order_shipped_status") . " SET `shipped`='1',`tracking_number`='" . $data['tracking_number'] . "' WHERE `order_id`=" . (int)$data['order_id']);
        } else {
            $this->db->query("INSERT INTO " . $this->db->table("pirateship_integration_order_shipped_status") . " SET `order_id`=" . (int)$data['order_id'].",`shipped`='1',`tracking_number`='" . $data['tracking_number']."'");
        }
    }

    /**
     * @param $oder_id
     * @return mixed
     * @throws AException
     */
    public function getStatus($oder_id) {
        $query = $this->db->query("SELECT COUNT(*) as `exist` FROM ".$this->db->table("pirateship_integration_order_shipped_status")." WHERE `order_id`=".(int)$order_id);
        return $query->row['exist'];
    }

    public function getStoreId() {
        $store_id = (int)$this->config->get('config_store_id');
        if (has_value($this->request->get_or_post('store_id'))) {
            $store_id = (int)$this->request->get_or_post('store_id');
        } else {
            if ($this->session->data['current_store_id']) {
                $store_id = (int)$this->session->data['current_store_id'];
            }
        }
        return $store_id;
    }

    /**
     * @param int $order_id
     * @param array $data
     *
     * @throws AException
     */
    public function addOrderHistory($order_id, $data) {
        $this->db->query(
            "UPDATE `".$this->db->table("orders")."`
            SET order_status_id = '".(int) $data['order_status_id']."',
                date_modified = NOW()
            WHERE order_id = '".(int) $order_id."'"
        );

        if ($data['append']) {
            $this->db->query(
                "INSERT INTO ".$this->db->table("order_history")."
                SET order_id = '".(int) $order_id."',
                    order_status_id = '".(int) $data['order_status_id']."',
                    notify = '".(isset($data['notify']) ? (int) $data['notify'] : 0)."',
                    comment = '".$this->db->escape($data['comment'])."',
                    date_added = NOW()"
            );
        }

        /*
         * Send Email with merchant comment.
         * Note IM-notification not needed here.
         * */
        if ($data['notify']) {
            $order_query = $this->db->query(
                "SELECT *, os.name AS status
                FROM `".$this->db->table("orders")."` o
                LEFT JOIN ".$this->db->table("order_statuses")." os 
                    ON (o.order_status_id = os.order_status_id AND os.language_id = o.language_id)
                LEFT JOIN ".$this->db->table("languages")." l 
                    ON (o.language_id = l.language_id)
                WHERE o.order_id = '".(int) $order_id."'"
            );

            if ($order_query->num_rows) {
                //load language specific for the order in admin section
                $language = new ALanguage(Registry::getInstance(), $order_query->row['code'], 1);
                $language->load($order_query->row['filename']);
                $language->load('mail/order');

                $this->load->model('setting/store');

                //send link to order only for registered customers
                $invoice = '';
                if ($order_query->row['customer_id']) {
                    $invoice = html_entity_decode(
                            $order_query->row['store_url'].'index.php?rt=account/invoice&order_id='.$order_id,
                            ENT_QUOTES, 'UTF-8'
                        )."\n\n";
                } //give link on order page for quest
                elseif ($this->config->get('config_guest_checkout') && $order_query->row['email']) {
                    $enc = new AEncryption($this->config->get('encryption_key'));
                    $order_token = $enc->encrypt($order_id.'::'.$order_query->row['email']);
                    $invoice = html_entity_decode(
                            $order_query->row['store_url'].'index.php?rt=account/invoice&ot='.$order_token,
                            ENT_QUOTES,
                            'UTF-8'
                        )
                        ."\n\n";
                }

                if ($this->dcrypt->active) {
                    $customer_email =
                        $this->dcrypt->decrypt_field($order_query->row['email'], $order_query->row['key_id']);
                } else {
                    $customer_email = $order_query->row['email'];
                }

                $this->data['mail_template_data'] = [
                    'store_name'        => $order_query->row['store_name'],
                    'order_id'          => $order_id,
                    'order_date_added'        => dateISO2Display(
                        $order_query->row['date_added'],
                        $language->get('date_format_short')
                    ),
                    'order_status' => $order_query->row['status'],
                    'invoice'           => $invoice,
                    'comment'           => $data['comment']
                        ?
                        html_entity_decode(
                            $data['comment'],
                            ENT_QUOTES, 'UTF-8'
                        )
                        : '',
                ];
                $this->extensions->hk_ProcessData($this, 'order_status_notify');
                $mail = new AMail($this->config);
                $mail->setTo($customer_email);
                $mail->setFrom($this->config->get('store_main_email'));
                $mail->setSender($order_query->row['store_name']);
                $mail->setTemplate('admin_order_status_notify', $this->data['mail_template_data']);
                $mail->send();

                //send IMs except emails.
                //TODO: add notifications for guest checkout
                $language->load('common/im');
                $invoice_url = $order_query->row['store_url'].'index.php?rt=account/invoice&order_id='.$order_id;
                //disable email protocol to prevent duplicates emails
                $this->im->removeProtocol('email');

                if ($order_query->row['customer_id']) {
                    $message_arr = [
                        0 => [
                            'message' => sprintf(
                                $language->get('im_order_update_text_to_customer'),
                                $invoice_url,
                                $order_id,
                                html_entity_decode($order_query->row['store_url'].'index.php?rt=account/account')
                            ),
                        ],
                    ];
                    $this->im->sendToCustomer($order_query->row['customer_id'], 'order_update', $message_arr);
                } else {
                    $message_arr = [
                        0 => [
                            'message' => sprintf(
                                $language->get('im_order_update_text_to_guest'),
                                $invoice_url,
                                $order_id,
                                html_entity_decode($invoice_url)
                            ),
                        ],
                    ];
                    $this->im->sendToGuest($order_id, $message_arr);
                }
                //turn email-protocol back
                $this->im->addProtocol('email');
            }
        }
    }
}