<?php

if (!defined('DIR_CORE')) {
    header('Location: static_pages/');
}

class ModelSaleIncompleteOrder extends Model {
    protected $timeout;

    public function __construct($registry) {
        parent::__construct($registry);
        $this->timeout = (int)$this->config->get('incomplete_order_timeout') ?: 48;
    }

    public function getHistory($order_id) {
        $result = $this->db->query(
            "SELECT *
            FROM ".$this->db->table('incomplete_order_history')."
            WHERE order_id = '".(int) $order_id."'"
        );

        if ($result->num_rows) {
            return $result->row;
        } else {
            return false;
        }
    }

    public function findSimilarOrders($order_id) {
        /** @var ModelSaleOrder $mdl */
        $mdl = $this->load->model('sale/order');
        $order = $mdl->getOrder($order_id);
        // check for completed orders placed around the same time
        $result = $this->db->query(
            "SELECT order_id, date_added
            FROM ".$this->db->table("orders")."
            WHERE email = '".$this->db->escape($order['email'])."'
                    AND order_status_id != 0
                    AND ( date_added >= ('".$this->db->escape($order['date_added'])."' + INTERVAL ".$this->timeout." HOUR ) )"
        );
        if ($result->num_rows) {
            return true;
        }

        // check for other incomplete orders placed around the same time that have already been e-mailed
        $result = $this->db->query(
            "SELECT o.order_id
            FROM ".$this->db->table("orders")." o
            LEFT JOIN ".$this->db->table("incomplete_order_history")." ioh ON (o.order_id = ioh.order_id)
            WHERE email = '".$this->db->escape($order['email'])."'
                AND order_status_id = 0
                AND o.date_added >= ('".$this->db->escape($order['date_added'])."' - INTERVAL ".$this->timeout." HOUR)
                        AND ioh.emailed = '1'"
        );

        return ($result->num_rows);
    }

    public function addHistory($order_id, $testing = false) {
        if ($this->getHistory($order_id)) {
            $this->db->query(
                "UPDATE ".$this->db->table('incomplete_order_history')."
                SET emailed = '1',
                    date_added = NOW()
                WHERE order_id = '".(int) $order_id."'"
            );
        } else {
            if ($testing) {
                if (!isset($this->session->data['temporary_incomplete_order_ids'])) {
                    $this->session->data['temporary_incomplete_order_ids'] = [];
                }
                $this->session->data['temporary_incomplete_order_ids'][] = $order_id;
            }
            $this->db->query(
                "INSERT INTO ".$this->db->table('incomplete_order_history')."
                SET order_id = '".(int) $order_id."', emailed = '1', date_added = NOW()"
            );
        }
    }

    public function removeHistory($order_id) {
        $this->db->query(
            "DELETE FROM ".$this->db->table('incomplete_order_history')."
            WHERE order_id = '".(int) $order_id."'"
        );
    }

    public function getOrder($order_id) {
        $order_query = $this->db->query(
            "SELECT *
            FROM ".$this->db->table("orders")."
            WHERE order_id = '".(int) $order_id."'"
        );

        if ($order_query->num_rows) {
            return [
                'order_id'          => $order_query->row['order_id'],
                'store_id'          => $order_query->row['store_id'],
                'store_name'        => $order_query->row['store_name'],
                'store_url'         => $order_query->row['store_url'],
                'customer_id'       => $order_query->row['customer_id'],
                'customer_group_id' => $order_query->row['customer_group_id'],
                'firstname'         => $order_query->row['firstname'],
                'lastname'          => $order_query->row['lastname'],
                'email'             => $order_query->row['email'],
                'total'             => $order_query->row['total'],
                'language_id'       => $order_query->row['language_id'],
                'order_status_id'   => $order_query->row['order_status_id'],
                'date_added'        => $order_query->row['date_added'],
                'ip'                => $order_query->row['ip'],
            ];
        } else {
            return false;
        }
    }

    /**
     * @return mixed
     * @throws AException
     */
    public function getCompanies() {
        $store_id = $this->getStoreId();
        $sql = "SELECT DISTINCT o.payment_company AS company
                FROM ".$this->db->table("orders")." o
                LEFT JOIN ".$this->db->table('incomplete_order_history')." ioh 
                    ON (o.order_id = ioh.order_id)";
        $sql .= " WHERE o.order_status_id = '0' AND store_id='".$store_id."'
                    AND o.date_added < (CURRENT_TIMESTAMP - INTERVAL ".$this->timeout." HOUR)";
        $query = $this->db->query($sql);
        return $query->rows;
    }

    /**
     * @param $data
     * @return mixed
     * @throws AException
     */
    public function getOrders($data = []) {
        $store_id = $this->getStoreId();
        $sql = "SELECT o.order_id,
                    CONCAT(o.firstname, ' ', o.lastname) AS name,
                    o.customer_id,
                    o.payment_company AS company,
                    CASE WHEN ISNULL(ioh.order_id) THEN 0 ELSE ioh.emailed END as emailed,
                    ioh.date_added as emailed_date,
                    o.date_added,
                    o.total,
                    o.currency,
                    o.value
                FROM ".$this->db->table("orders")." o
                LEFT JOIN ".$this->db->table('incomplete_order_history')." ioh 
                    ON (o.order_id = ioh.order_id)";
        $sql .= " WHERE o.order_status_id = '0' AND store_id='".$store_id."'
                    AND o.date_added < (CURRENT_TIMESTAMP - INTERVAL ".$this->timeout." HOUR)";

        if ($data['filter_order_id']) {
            $sql .= " AND o.order_id = '".(int) $data['filter_order_id']."'";
        }

        if (!empty($data['filter_name'])) {
            $sql .= " AND CONCAT(o.firstname, ' ', o.lastname) LIKE '%".$this->db->escape($data['filter_name'])."%'";
        }

        if (!empty($data['filter_company'])) {
            if ($data['filter_company']==='empty') {
                $sql .= " AND o.payment_company = ''";
            } else {
                $sql .= " AND o.payment_company LIKE '%".$this->db->escape($data['filter_company'])."%'";
            }
        }

        if (has_value($data['filter_account_status'])) {
            if ($data['filter_account_status'] === '1') {
                $sql .= " AND o.customer_id <> 0";
            } elseif ($data['filter_account_status'] === '0') {
                $sql .= " AND o.customer_id = 0";
            }
        }

        if (isset($data['filter_date_added']) && !empty($data['filter_date_added'])) {
            $sql .= " AND DATE(o.date_added) = DATE('".$this->db->escape($data['filter_date_added'])."')";
        }

        if (!empty($data['filter_date_range'])) {
            $sql .= " AND DATE(o.date_added) >= DATE_SUB(CURDATE(), INTERVAL "
                        .$this->db->escape($data['filter_date_range']).")";
        }

        if ($data['filter_total']) {
            $data['filter_total'] = (float) str_replace(',', '.', $data['filter_total']);
            $data['filter_total'] = floor($data['filter_total']);
            $currencies = $this->currency->getCurrencies();
            $temp = $temp2 = [$data['filter_total']];
            foreach ($currencies as $currency1) {
                foreach ($currencies as $currency2) {
                    if ($currency1['code'] != $currency2['code']) {
                        $temp[] = round(
                            $this->currency->convert($data['filter_total'], $currency1['code'], $currency2['code']), 0
                        );
                        $temp2[] = floor(
                            $this->currency->convert($data['filter_total'], $currency1['code'], $currency2['code'])
                        );
                    }
                }
            }
            $sql .= " AND ( ROUND(o.total,0) IN  (".implode(",", $temp).")
                            OR ROUND(CAST(o.total as DECIMAL(15,4)) * CAST(o.value as DECIMAL(15,4)),0) 
                                IN  (".implode(",", $temp).")
                            OR TRUNCATE(CAST(o.total as DECIMAL(15,4)) * CAST(o.value as DECIMAL(15,4)),0) 
                                IN  (".implode(",", $temp2).")) ";
        }

        if (has_value($data['filter_emailed'])) {
            if ($data['filter_emailed']) {
                $sql .= " AND ioh.emailed = '1'";
            } else {
                $sql .= " AND COALESCE(ioh.emailed,0) = 0";
            }
        }

        $sort_data = [
            'o.order_id',
            'name',
            'account_status',
            'company',
            'o.date_added',
            'o.total',
            'emailed',
        ];

        if (isset($data['sort']) && in_array($data['sort'], $sort_data)) {
            if ($data['sort']==='account_status') {
                $sql .= " ORDER BY o.customer_id";
            } else {
                $sql .= " ORDER BY ".$data['sort'];
            }
        } else {
            $sql .= " ORDER BY o.order_id";
        }

        if (isset($data['order']) && ($data['order'] == 'DESC')) {
            $sql .= " DESC";
        } else {
            $sql .= " ASC";
        }

        if (isset($data['start']) || isset($data['limit'])) {
            if ($data['start'] < 0) {
                $data['start'] = 0;
            }

            if ($data['limit'] < 1) {
                $data['limit'] = 20;
            }

            $sql .= " LIMIT ".(int) $data['start'].",".(int) $data['limit'];
        }

        $query = $this->db->query($sql);

        return $query->rows;
    }

    public function getOrderProducts($order_id) {
        $query = $this->db->query(
            "SELECT *
            FROM ".$this->db->table('order_products')."
            WHERE order_id = '".(int) $order_id."'"
        );

        return $query->rows;
    }

    public function getOrderOptions($order_id, $order_product_id) {
        $query = $this->db->query(
            "SELECT *
            FROM ".$this->db->table('order_options')."
            WHERE order_id = '".(int) $order_id."'
                AND order_product_id = '".(int) $order_product_id."'"
        );

        return $query->rows;
    }

    public function getTotalOrders($data = []) {
        $store_id = $this->getStoreId();
        $sql = "SELECT COUNT(*) AS total
                FROM ".$this->db->table("orders")." o
                LEFT JOIN ".$this->db->table("incomplete_order_history")." ioh ON (o.order_id = ioh.order_id)";

        $sql .= " WHERE order_status_id = '0' AND store_id='".$store_id."'
                    AND 
                    o.date_added < (CURRENT_TIMESTAMP - INTERVAL ".$this->timeout." HOUR)";

        if (isset($data['filter_order_id']) && !is_null($data['filter_order_id'])) {
            $sql .= " AND o.order_id = '".(int) $data['filter_order_id']."'";
        }

        if (!empty($data['filter_name'])) {
            $sql .= " AND CONCAT(firstname, ' ', lastname) LIKE '%".$this->db->escape($data['filter_name'])."%'";
        }

        if (isset($data['filter_date_added']) && !is_null($data['filter_date_added'])) {
            $sql .= " AND DATE(o.date_added) = DATE('".$this->db->escape($data['filter_date_added'])."')";
        }

        if (!empty($data['filter_total'])) {
            $sql .= " AND total = '".(float) $data['filter_total']."'";
        }

        if (isset($data['filter_emailed']) && !is_null($data['filter_emailed'])) {
            if ($data['filter_emailed']) {
                $sql .= " AND ioh.emailed = '1'";
            } else {
                $sql .= " AND COALESCE(ioh.emailed, 0) = 0";
            }
        }

        $query = $this->db->query($sql);
        return $query->row['total'];
    }

    public function cancelOrder() {
        if ((int) $this->config->get('incomplete_order_expiration_days')) {
            $query = $this->db->query(
                "SELECT order_id
                    FROM ".$this->db->table("orders")."
                    WHERE date_modified < '"
                .date(
                    'Y-m-d',
                    strtotime('-'.(int) $this->config->get('incomplete_order_expiration_days').' days')
                )."' AND order_status_id = '0'"
            );
            foreach ($query->rows as $result) {
                $this->cancelSingleOrder($result['order_id']);
            }
        }
    }

    public function cancelSingleOrder($order_id) {
        $order_id = (int) $order_id;
        if (!$order_id) {
            return false;
        }
        $this->db->query("UPDATE `".$this->db->table("orders")."` SET `order_status_id` = '7' WHERE order_id = '".$order_id."'");
    }

    /**
     * @param $section
     * @return mixed
     * @throws AException
     */
    public function getDetails($section) {
        $store_id = $this->getStoreId();
        $sql="SELECT mail.sender_name,mail.sender_email,mail.subject,mail.content,mail.allowed_placeholders, sigs.content as signature, notify_admin
              FROM ". $this->db->table('email_editor_templates') . " mail 
              LEFT JOIN ". $this->db->table('email_editor_signatures') . " sigs on sigs.email_signature_id=mail.email_signature_id
              WHERE mail.section='".$section."' AND mail.store_id=".$store_id;
        $result=$this->db->query($sql);
        return $result->row;
    }

    public function getStoreId() {
        $store_id = (int) $this->config->get('config_store_id');
        if (has_value($this->request->get_or_post('store_id'))) {
            $store_id = (int) $this->request->get_or_post('store_id');
        } else {
            if ($this->session->data['current_store_id']) {
                $store_id = (int) $this->session->data['current_store_id'];
            }
        }
        return $store_id;
    }
}