<?php
/** @noinspection PhpMultipleClassDeclarationsInspection */
/*------------------------------------------------------------------------------

  For Abante Cart, E-commerce Solution
  http://www.AbanteCart.com

  Copyright (c) 2014-2024 We Hear You 2, Inc.  (WHY2)

------------------------------------------------------------------------------*/

use ReCaptcha\ReCaptcha;
/**
 * Load form data, render output
 *
 * Class AForm
 *
 * @property ALayout $layout
 * @property ACache $cache
 * @property ADB $db
 * @property AConfig $config
 * @property AHtml $html
 * @property ASession $session
 * @property ARequest $request
 * @property ALoader $load
 * @property ALanguageManager $language
 * @property ModelLocalisationCountry $model_localisation_country
 *
 */
class AHcaptchaForm extends AForm {
    /**
     * @var Registry
     */
    protected $registry;
    /**
     * @var array $form - form data array
     * Array (
     *      [form_id]
     *      [form_name]
     *      [controller]
     *      [success_page]
     *      [status]
     *      [description] )
     */
    protected $form;
    /**
     * @var array $fields - array of form fields
     * Array
     *   (
     *       [field_id]
     *       [form_id]
     *       [field_name]
     *       [element_type]
     *       [sort_order]
     *       [required]
     *       [status]
     *       [name]
     *       [description]
     *       [value]
     *   )
     */
    protected $fields;
    /**
     * @var $groups - fields groups
     */
    protected $groups;
    /**
     * @var $page_id - current page id
     */
    public $page_id;
    /**
     * @var $errors - field_name -=> error
     */
    protected $errors;
    /**
     * @var $form_edit_action - ( ST = standard,  HS = highlight save,  HT = highlight )
     */
    protected $form_edit_action;

    /**
     * return field html
     *
     * @param array $data - array with field data
     *
     * @return object  - AHtml form element
     * @throws AException
     */
    public function getFieldHtml($data)
    {
        $data['form'] = $this->form['form_name'] ?? '';

        if ($data['type'] == 'form') {
            $data['javascript'] = $this->addFormJs();
        }

        return HtmlHcaptchaElementFactory::create($data);
    }

    /**
     * Process and render the form with HTML output.
     *
     * @param bool $fieldsOnly
     *
     * @return string html
     * @throws AException
     */
    public function getFormHtml($fieldsOnly = false)
    {
        // if no form was loaded return empty string
        if (empty($this->form)) {
            return '';
        }

        $fields_html = [];
        $view = new AView($this->registry, 0);

        $log=new ALog(DIR_SYSTEM.'logs/error.txt');
        foreach ($this->fields as $field) {
            //check for enabled recaptcha instead of default captcha
            if ($this->config->get('config_recaptcha_site_key') && $field['element_type'] == 'K') {
                $field['element_type'] = 'J';
            } elseif (!$this->config->get('config_recaptcha_site_key') && $field['element_type'] == 'K') {
                $field['element_type'] = 'Q';
            }

            //build data array for each field HTML template
            $data = [
                'type'     => HtmlHcaptchaElementFactory::getElementType($field['element_type']),
                'name'     => $field['field_name'],
                'form'     => $this->form['form_name'],
                'attr'     => $field['attributes'],
                'required' => $field['required'],
                'value'    => $field['value'],
                'options'  => $field['options'],
            ];

            //populate customer entered values from session (if present)
            if (is_array($this->session->data['custom_form_'.$this->form['form_id']])) {
                $data['value'] = $this->session->data['custom_form_'.$this->form['form_id']][$field['field_name']];
            }

            //custom data based on the HTML element type
            switch ($data['type']) {
                case 'multiselectbox' :
                case 'checkboxgroup' :
                    $data['name'] .= '[]';
                    break;
                case 'captcha' :
                case 'hcaptcha' :
                    $data['site_key'] = $this->config->get('hcaptcha_integration_site_key');
                    $data['theme'] = $this->config->get('hcaptcha_integration_theme');
                    $data['size'] = $this->config->get('hcaptcha_integration_widget_size');
                    $data['language_code'] = $this->language->getLanguageCode();
                    break;
                case 'recaptcha' :
                    $data['recaptcha_site_key'] = $this->config->get('config_recaptcha_site_key');
                    $data['language_code'] = $this->language->getLanguageCode();
                    break;
            }
            $item = HtmlHcaptchaElementFactory::create($data);

            switch ($data['type']) {
                case 'IPaddress' :
                case 'hidden' :
                    $fields_html[$field['field_id']] = $item->getHtml();
                    break;
                default:
                    $view->batchAssign(
                        [
                            'element_id'  => $item->element_id,
                            'type'        => $data['type'],
                            'title'       => $field['name'],
                            'description' => (!empty($field['description']) ? $field['description'] : ''),
                            'error'       => (!empty($this->errors[$field['field_name']])
                                ? $this->errors[$field['field_name']]
                                : ''),
                            'item_html'   => $item->getHtml(),
                        ]
                    );
                    $fields_html[$field['field_id']] = $view->fetch('extensions/hcaptcha_integration/storefront/view/'.$this->config->get('config_storefront_template').'/template/form/form_field.tpl');
            }
        }

        $output = '';
        if (!empty($this->groups)) {
            foreach ($this->groups as $group) {
                $view->batchAssign(
                    [
                        'group'       => $group,
                        'fields_html' => $fields_html,
                    ]
                );
                $output .= $view->fetch('form/form_group.tpl');
            }
        } else {
            $view->batchAssign(['fields_html' => $fields_html]);
            $output .= $view->fetch('form/form_no_group.tpl');
        }

        // add submit button and form open/close tag
        if (!$fieldsOnly) {
            $data = [
                'type' => 'submit',
                'form' => $this->form['form_name'],
                'name' => $this->language->get('button_submit'),
            ];
            $submit = HtmlHcaptchaElementFactory::create($data);

            $data = [
                'type'   => 'form',
                'name'   => $this->form['form_name'],
                'attr'   => ' class="form" ',
                'action' => $this->html->getSecureURL(
                    $this->form['controller'],
                    '&form_id='.$this->form['form_id'],
                    true
                ),
            ];
            $form_open = HtmlHcaptchaElementFactory::create($data);
            $form_close = $view->fetch('form/form_close.tpl');

            $js = $this->addFormJs();

            $view->batchAssign(
                [
                    'description'  => $this->form['description'],
                    'form'         => $output,
                    'form_open'    => $js.$form_open->getHtml(),
                    'form_close'   => $form_close,
                    'submit'       => $submit,
                    'button_reset' => $this->language->get('button_reset'),
                ]
            );
            $output = $view->fetch('form/form.tpl');
        }

        return $output;
    }

    /**
     * method for validation of data based on form fields requirements
     *
     * @param array $data - usually it's a $_POST
     *
     * @return array - array with error text for each of invalid field data
     * @throws AException
     */
    public function validateFormData($data = [])
    {
        $errors = [];
        $this->_loadFields();
        $this->load->language('checkout/cart'); // load language for file upload text errors

        foreach ($this->fields as $field) {
            // for multivalue required fields
            if (in_array($field['element_type'], HtmlHcaptchaElementFactory::getMultivalueElements())
                && !$data[$field['field_name']]
                && $field['required'] == 'Y'
            ) {
                $errors[$field['field_name']] = $field['name'].' '.$this->language->get('text_field_required');
            }
            // for required string values
            if ($field['required'] == 'Y' && !in_array($field['element_type'], ['K', 'J', 'U'])) {
                if (!is_array($data[$field['field_name']])) {
                    $data[$field['field_name']] = trim($data[$field['field_name']]);
                    //if empty string!
                    if ($data[$field['field_name']] == '') {
                        $errors[$field['field_name']] = $field['name'].' '.$this->language->get('text_field_required');
                    }
                } else {
                    // if empty array
                    if (!$data[$field['field_name']]) {
                        $errors[$field['field_name']] = $field['name'].' '.$this->language->get('text_field_required');
                    }
                }
            }
            // check by regexp
            if (has_value($field['regexp_pattern'])) {
                if (!is_array($data[$field['field_name']])) { //for string value
                    if (!preg_match($field['regexp_pattern'], $data[$field['field_name']])) {
                        // show error only for field with value or required
                        if (($data[$field['field_name']] && $field['required'] != 'Y') || $field['required'] == 'Y') {
                            $errors[$field['field_name']] .= ' '.$field['error_text'];
                        }
                    }
                } else {
                    // for array's values
                    foreach ($data[$field['field_name']] as $dd) {
                        if (!preg_match($field['regexp_pattern'], $dd)) {
                            if (($dd && $field['required'] != 'Y') || $field['required'] == 'Y') {
                                $errors[$field['field_name']] .= ' '.$field['error_text'];
                            }
                            break;
                        }
                    }
                }
            }

            //for captcha or recaptcha
            if ($field['element_type'] == 'K' || $field['element_type'] == 'J') {
                if ($this->config->get('config_recaptcha_secret_key')) {
                    $recaptcha = new ReCaptcha($this->config->get('config_recaptcha_secret_key'));
                    $resp = $recaptcha->verify($data['g-recaptcha-response'], $this->request->getRemoteIP());
                    if (!$resp->isSuccess() && $resp->getErrorCodes()) {
                        $errors[$field['field_name']] = $this->language->get('error_captcha');
                    }
                } else {
                    if (!isset($this->session->data['captcha'])
                        || ($this->session->data['captcha'] != $data[$field['field_name']])
                    ) {
                        $errors[$field['field_name']] = $this->language->get('error_captcha');
                    }
                }
            }

            // for file
            if ($field['element_type'] == 'U'
                && ($this->request->files[$field['field_name']]['tmp_name'] || $field['required'] == 'Y')
            ) {
                $fm = new AFile();
                $file_path_info = $fm->getUploadFilePath(
                    $data['settings']['directory'],
                    $this->request->files[$field['field_name']]['name']
                );
                $file_data = [
                    'name'     => $file_path_info['name'],
                    'path'     => $file_path_info['path'],
                    'type'     => $this->request->files[$field['field_name']]['type'],
                    'tmp_name' => $this->request->files[$field['field_name']]['tmp_name'],
                    'error'    => $this->request->files[$field['field_name']]['error'],
                    'size'     => $this->request->files[$field['field_name']]['size'],
                ];

                $file_errors = $fm->validateFileOption($field['settings'], $file_data);

                if ($file_errors) {
                    $errors[$field['field_name']] .= implode(' ', $file_errors);
                }
            }
        }

        return $errors;
    }
}