<?php
/*------------------------------------------------------------------------------

  For Abante Cart, E-commerce Solution
  http://www.AbanteCart.com

  Copyright (c) 2014-2024 We Hear You 2, Inc.  (WHY2)

------------------------------------------------------------------------------*/

if (!defined('DIR_CORE')) {
    header('Location: static_pages/');
}

/**
 * Class ModelSaleCustomerPreferences
 */
class ModelSaleCustomerPreferences extends Model {
    /**
     * @param int $customer_group_id
     * @param int $language_id
     *
     * @return array
     * @throws AException
     */
    public function getPreferencesForGroup($customer_group_id, $language_id)
    {
        $language_id = $language_id ? : $this->language->getContentLanguageID();
        $results = $this->db->query(
            "SELECT
                cag.customer_preferences_id,
                cag.customer_group_id,
                cad.name, cad.error_text
            FROM ".$this->db->table("customer_preferences_to_groups")." cag
            LEFT JOIN ".$this->db->table("customer_preferences")." ca ON ca.customer_preferences_id = cag.customer_preferences_id
            INNER JOIN ".$this->db->table("customer_preferences_descriptions")." cad ON ca.customer_preferences_id = cad.customer_preferences_id
            WHERE
                cag.customer_group_id = ".(int) $customer_group_id."
                AND cad.language_id = ".(int) $language_id
        );

        return $results->rows;
    }

    public function getPreferencesForCustGroup($customer_group_id, $language_id, $show_in = '') {

        if ( $show_in != '' && !in_array($show_in, $this->show_in) ) {
            return [];
        }

        $language_id = (int) $language_id;
        $language_id = $language_id ? : $this->config->get('storefront_language_id');

        $where = "gad.language_id = " . (int) $language_id . "
				AND cag.visible_by_customer = 1
				AND cag.status = 1";

        if ( $show_in != '' ) {
            $where .= " AND cag.show_" . $show_in . " = 1";
        }

        $where .= " AND cag.customer_group_id = " . (int) $customer_group_id;

        $sql = "SELECT  cag.customer_preferences_id,
                        cag.customer_group_id,
                        cag.editable_by_customer,
                        cag.visible_by_customer,
                        gad.name,
                        gad.error_text,
                        ga.element_type,
                        ga.sort_order,
                        ga.required,
                        ga.regexp_pattern,
                        ga.settings
                FROM ".$this->db->table('customer_preferences_to_groups')." cag
                INNER JOIN ".$this->db->table('customer_preferences_descriptions')." gad
                    ON cag.customer_preferences_id = gad.customer_preferences_id 
                        AND gad.language_id = ".(int) $language_id."
                INNER JOIN ".$this->db->table('customer_preferences')." ga 
                    ON cag.customer_preferences_id = ga.customer_preferences_id
               WHERE " . $where . "
                ORDER BY ga.sort_order ASC";

        $results = $this->db->query($sql);
        $output = (array) $results->rows;
        foreach ($output as &$row) {
            $row['values'] = $this->getPreferencesValues($row['customer_preferences_id'], $language_id);
            $row['selected_values'] = [];
            $row['settings'] = $row['settings'] ? unserialize($row['settings']) : [];
        }
        unset($row);
        return $output;
    }


    /**
     * @param $customer_id
     * @return void
     * @throws AException
     */
    public function deleteCustPreferences($customer_id) {
        $this->db->query(
            "DELETE FROM " . $this->db->table('customer_preferences_values') . "
                 WHERE customer_id = '" . (int)$customer_id . "';"
        );
    }

    /**
     * @param $customer_id
     * @param $data
     * @return bool
     * @throws AException
     */
    public function addPreferencesToCustomer($customer_id, $data = []) {
        if (!is_array($data) || !(int) $customer_id) {
            return false;
        }

        $this->db->query(
            "INSERT INTO " . $this->db->table('customer_preferences_values') . "
                        (customer_id, customer_preferences_id, customer_preferences_value)
                    VALUES (" . (int)$customer_id . ", " . (int)$data['customer_preferences_id'] . ", '" . addslashes($this->db->escape($data['value'])) . "');"
        );
        return true;
    }

    /**
     * @param int $customer_preferences_id
     * @param int $customer_group_id
     *
     * @return array
     * @throws AException
     */
    public function getPreferencesInfo($customer_preferences_id, $customer_group_id) {
        $language_id = (int)$this->language->getContentLanguageID();
        $sql = "SELECT
                        cag.customer_group_id,
                        cag.editable_by_customer,
                        cag.visible_by_customer,
                        cag.show_shipping,
                       cag.show_checkout_confirm,
                       cag.show_fast_checkout,
                       cag.show_admin_invoice,
                       cag.show_invoice,
                       cag.show_confirmation_email,
                        cag.status,
                        gad.name,
                        gad.error_text,
                        ca.customer_preferences_id,
                        ca.element_type,
                        ca.sort_order,
                        ca.settings,
                        ca.required,
                        ca.regexp_pattern
                    FROM ".$this->db->table('customer_preferences_to_groups')." cag
                    LEFT JOIN ".$this->db->table('customer_preferences')." ca ON cag.customer_preferences_id = ca.customer_preferences_id
                    LEFT JOIN ".$this->db->table('customer_preferences_descriptions')." gad
                        ON ( cag.customer_preferences_id = gad.customer_preferences_id 
                            AND gad.language_id = '".$language_id."')
                    WHERE ca.customer_preferences_id = ".(int) $customer_preferences_id."
                        ".((int) $customer_group_id ? " AND cag.customer_group_id='".(int) $customer_group_id."'" : "")."
                    ORDER BY ca.sort_order ASC";
        $results = $this->db->query($sql);
        $output = $results->rows;
        foreach ($output as &$row) {
            $settings = unserialize($row['settings']);
            $row['settings'] = [];
            $row['settings'] = $settings;
            $row['values'] = $this->getPreferencesValues($row['customer_preferences_id'], $language_id);
        }
        return $output;
    }

    /**
     * @param int $attribute_id
     * @param int $language_id
     *
     * @return array
     * @throws AException
     */
    public function getPreferencesValues($attribute_id, $language_id) {
        $language_id = !$language_id ? $this->language->getContentLanguageID() : $language_id;
        $sql = "SELECT gav.customer_preferences_value_id, gav.sort_order,gav.default, gavd.value
                FROM ".$this->db->table('customer_preferences_values_to_groups')." gav
                INNER JOIN ".$this->db->table('customer_preferences_value_descriptions_to_groups')." gavd
                    ON (gav.customer_preferences_value_id = gavd.customer_preferences_value_id 
                        AND gavd.language_id = ".(int) $language_id.")
                WHERE gav.customer_preferences_id = ".(int) $attribute_id."
                ORDER BY gav.sort_order ASC";
        $results = $this->db->query($sql);
        return $results->rows;
    }

    /**
     * @param int $customer_group_id
     * @param array $data
     *
     * @return bool
     * @throws AException
     */
    public function addPreferencesToGroup($customer_group_id, $data) {
        if (!$data || !$customer_group_id) {
            return null;
        }
        if ((int) $data['attribute_id']) {
            $sql = "INSERT INTO ".$this->db->table('customer_preferences_to_groups')."
                    SET
                    customer_preferences_id = ".(int) $data['customer_preferences_id'].",
                    customer_group_id = ".(int) $customer_group_id;

            if (isset($data['editable'])) {
                $edit = ((int) $data['editable'] > 0) ? 1 : 0;
                $sql .= ", editable_by_customer = ".$edit;
            }

            if (isset($data['visible'])) {
                $visible = ((int) $data['visible'] > 0) ? 1 : 0;
                $sql .= ", visible_by_customer = ".$visible;
            }

            if (isset($data['show_shipping'])) {
                $show_shipping = ((int) $data['show_shipping'] > 0) ? 1 : 0;
                $sql .= ", show_shipping = ".$show_shipping;
            }

            if (isset($data['show_checkout_confirm'])) {
                $show_checkout_confirm = ((int) $data['show_checkout_confirm'] > 0) ? 1 : 0;
                $sql .= ", show_checkout_confirm = ".$show_checkout_confirm;
            }

            if (isset($data['show_fast_checkout'])) {
                $show_fast_checkout = ((int) $data['show_fast_checkout'] > 0) ? 1 : 0;
                $sql .= ", show_fast_checkout = ".$show_fast_checkout;
            }

            if (isset($data['show_admin_invoice'])) {
                $show_admin_invoice = ((int) $data['show_admin_invoice'] > 0) ? 1 : 0;
                $sql .= ", show_admin_invoice = ".$show_admin_invoice;
            }

            if (isset($data['show_invoice'])) {
                $show_invoice = ((int) $data['show_invoice'] > 0) ? 1 : 0;
                $sql .= ", show_invoice = ".$show_invoice;
            }

            if (isset($data['show_confirmation_email'])) {
                $show_confirmation_email = ((int) $data['show_confirmation_email'] > 0) ? 1 : 0;
                $sql .= ", show_confirmation_email = ".$show_confirmation_email;
            }

            if (isset($data['status'])) {
                $status = ((int) $data['status'] > 0) ? 1 : 0;
            } else {
                $status = 0;
            }
            $sql .= ", status = ".$status;

            $this->db->query($sql);
            return true;
        }
        return false;
    }

    /**
     * Function delete ONLY assistance of customer attribute to customer group
     *
     * @param int $customer_preferences_id
     * @param int $customer_group_id
     *
     * @throws AException
     */
    public function removeCustomerPreferencesFromGroup($customer_preferences_id, $customer_group_id) {
        $this->db->query(
            "DELETE FROM ".$this->db->table('customer_preferences_to_groups')."
            WHERE customer_preferences_id = ".(int) $customer_preferences_id."
                AND customer_group_id = ".(int) $customer_group_id
        );

        $this->db->query(
            "DELETE FROM ".$this->db->table('customer_preferences_values_to_groups')."
            WHERE customer_preferences_id = ".(int) $customer_preferences_id." 
                AND customer_group_id = ".(int) $customer_group_id
        );
        $this->db->query(
            "DELETE FROM ".$this->db->table('customer_preferences_value_descriptions_to_groups')."
            WHERE customer_preferences_id = '.(int) $customer_preferences_id.' 
                AND customer_group_id = ".(int) $customer_group_id);
    }

    /**
     * @param int $customer_group_id
     * @param array $data
     *
     * @return bool|int
     * @throws AException
     */
    public function addCustomerPreferences($customer_group_id, $data) {

        if (!(int) $customer_group_id || !is_array($data)) {
            return false;
        }

        $attribute_manager = new AAttribute_Manager();
        if ((int) $data['global_attribute_id']) {
            $ga_info = $attribute_manager->getAttribute((int) $data['global_attribute_id']);
            if (!$ga_info) {
                return false;
            }
            $ga_info['attribute_descriptions'] = $attribute_manager->getAttributeDescriptions(
                (int) $data['global_attribute_id']
            );
            $data = array_merge($data, $ga_info);
        } else {
            $data['attribute_descriptions'][$this->language->getContentLanguageID()] = [
                'name'       => $data['name'],
                'error_text' => $data['error_text'],
            ];
        }

        $this->db->query(
            "INSERT INTO ".$this->db->table('customer_preferences')."
            SET `global_attribute_id` = '".(int) $data['global_attribute_id']."',
                `element_type` = '".$this->db->escape($data['element_type'])."',
                `sort_order` = '".(int) $data['sort_order']."',
                `required` = '".(int) $data['required']."',
                `settings` = '".$this->db->escape($data['settings'])."',
                `regexp_pattern` = '".$this->db->escape($data['regexp_pattern'])."'"
        );
        $customer_preferences_id = $this->db->getLastId();
        // assign customer attribute to customer group
        $this->db->query(
            "INSERT INTO ".$this->db->table('customer_preferences_to_groups')."
            SET `customer_preferences_id` = '".(int) $customer_preferences_id."',
                `customer_group_id` = '".(int) $customer_group_id."',
                `editable_by_customer` = '".(int) $data['editable_by_customer']."',
                `visible_by_customer` = '".(int) $data['visible_by_customer']."',
                `show_shipping` = '".(int) $data['show_shipping']."',
                `show_checkout_confirm` = '".(int) $data['show_checkout_confirm']."',
                `show_fast_checkout` = '".(int) $data['show_fast_checkout']."',
                `show_admin_invoice` = '".(int) $data['show_admin_invoice']."',
                `show_invoice` = '".(int) $data['show_invoice']."',
                `show_confirmation_email` = '".(int) $data['show_confirmation_email']."',
                `status` = '".(int) $data['status']."'"
        );

        // insert descriptions for used content language and translate
        $allowed_fields = ['name', 'error_text'];
        foreach ($data['attribute_descriptions'] as $lang_id => $values) {
            foreach ($values as $n => $v) {
                if (!in_array($n, $allowed_fields)) {
                    continue;
                }
                $attr_descriptions[$lang_id][$n] = $v;
            }
        }

        $this->language->replaceDescriptions(
            'customer_preferences_descriptions',
            ['customer_preferences_id' => (int) $customer_preferences_id],
            $attr_descriptions
        );
        if ((int) $data['global_attribute_id']) {
            $ga_values = $attribute_manager->getAttributeValues($data['global_attribute_id']);
            foreach ($ga_values as $ga_value) {
                $attribute_value_id = $this->addCustomerPreferencesValue(
                    [
                        'customer_preferences_id' => $customer_preferences_id,
                        'customer_group_id'     => $customer_group_id,
                        'editable_by_customer'  => (int) $data['editable_by_customer'],
                        'visible_by_customer'   => (int) $data['visible_by_customer'],
                        'show_shipping'   => (int) $data['show_shipping'],
                        'show_checkout_confirm'   => (int) $data['show_checkout_confirm'],
                        'show_fast_checkout'   => (int) $data['show_fast_checkout'],
                        'show_admin_invoice'   => (int) $data['show_admin_invoice'],
                        'show_invoice'   => (int) $data['show_invoice'],
                        'show_confirmation_email'   => (int) $data['show_confirmation_email'],
                        'sort_order'            => (int) $ga_value['sort_order'],
                    ]
                );

                $ga_values_descriptions = $attribute_manager->getAttributeValueDescriptions(
                    $ga_value['attribute_value_id']
                );
                foreach ($ga_values_descriptions as $lang_id => $ga_val) {
                    $descriptions[$lang_id] = ['value' => $ga_val];
                }
                $all_languages = $this->language->getActiveLanguages();
                if (!$descriptions[$this->language->getDefaultLanguageID()]) {
                    $default_descriptions = current($descriptions);
                } else {
                    $default_descriptions = $descriptions[$this->language->getDefaultLanguageID()];
                }

                foreach ($all_languages as $lang) {
                    if (!in_array($lang['language_id'], array_keys($descriptions))) {
                        $descriptions[$lang['language_id']] = $default_descriptions;
                    }
                }

                $this->language->replaceDescriptions(
                    'customer_preferences_value_descriptions_to_groups',
                    [
                        'customer_preferences_value_id' => $attribute_value_id,
                        'customer_preferences_id'       => (int) $customer_preferences_id,
                        'customer_group_id'           => (int) $customer_group_id,
                    ],
                    $descriptions
                );
            }
        } else {
            //add empty option value for single value attributes
            $elements_with_options = HtmlElementFactory::getElementsWithOptions();
            if (!in_array($data['element_type'], $elements_with_options)) {
                $attr_val_id = $this->addCustomerPreferencesValue(
                    [
                        'customer_preferences_id' => $customer_preferences_id,
                        'customer_group_id'     => (int) $customer_group_id,
                    ]
                );
                $all_languages = $this->language->getActiveLanguages();
                $descr = [];
                foreach ($all_languages as $lang) {
                    $descr[$lang['language_id']] = ['value' => $data['name']];
                }
                $this->language->replaceDescriptions(
                    'customer_preferences_value_descriptions_to_groups',
                    [
                        'customer_preferences_value_id' => (int) $attr_val_id,
                        'customer_preferences_id'       => $customer_preferences_id,
                        'customer_group_id'           => $customer_group_id,
                    ],
                    $descr
                );
            }
        }
        return $customer_preferences_id;
    }

    /**
     * @param array $data
     *
     * @return bool|int
     * @throws AException
     */
    public function addCustomerPreferencesValue($data = []) {
        if (empty($data['customer_preferences_id']) || empty($data['customer_group_id'])) {
            return false;
        }
        $sql = "INSERT INTO `".$this->db->table('customer_preferences_values_to_groups')."`
                SET `customer_preferences_id` = '".(int) $data['customer_preferences_id']."',
                    `customer_group_id` = '".(int) $data['customer_group_id']."',
                    `sort_order` = '".(int) $data['sort_order']."',
                    `default` ='".(int)$data['default']."'";
        $this->db->query($sql);
        return $this->db->getLastId();
    }

    /**
     * @param int $customer_preferences_id
     * @param array $data
     *
     * @throws AException
     */
    public function updatePreferences($customer_preferences_id, $data = []) {
        //Note: update is done per 1 language
        $language_id = $this->session->data['content_language_id'];
        $fields = [
            'required',
            'sort_order',
            'settings',
            'regexp_pattern',
        ];

        if (has_value($data['settings'])) {
            $data['settings'] = serialize($data['settings']);
        }

        $update = [];
        foreach ($fields as $f) {
            if (isset($data[$f])) {
                $update[] = "$f = '".$this->db->escape($data[$f])."'";
            }
        }

        if (!empty($update)) {
            $sql = "UPDATE ".$this->db->table('customer_preferences')."
                    SET ".implode(',', $update)."
                    WHERE customer_preferences_id = '".(int) $customer_preferences_id."'";
            $this->db->query($sql);
        }

        $sql = "UPDATE ".$this->db->table('customer_preferences_to_groups')."
                SET
                `status` = '".(int) $data['status']."',
                `editable_by_customer` = '".(int) $data['editable']."',
                `visible_by_customer` = '".(int) $data['visible']."',
                `show_shipping` = '".(int) $data['show_shipping']."',
                `show_checkout_confirm` = '".(int) $data['show_checkout_confirm']."',
                `show_fast_checkout` = '".(int) $data['show_fast_checkout']."',
                `show_admin_invoice` = '".(int) $data['show_admin_invoice']."',
                `show_invoice` = '".(int) $data['show_invoice']."',
                `show_confirmation_email` = '".(int) $data['show_confirmation_email']."'
                WHERE `customer_preferences_id` = '".(int) $customer_preferences_id."'
                        AND `customer_group_id` = '".(int) $data['customer_group_id']."'";
        $this->db->query($sql);

        $update = [];
        if (isset($data['name'])) {
            $update['name'] = $data['name'];
        }
        if (isset($data['error_text'])) {
            $update['error_text'] = $data['error_text'];
        }

        $this->language->replaceDescriptions(
            'customer_preferences_descriptions',
            ['customer_preferences_id' => (int) $customer_preferences_id],
            [$language_id => $update]
        );
        $this->clearCache();
    }

    /**
     * @param array $data
     *
     * @throws AException
     */
    public function updatePreferencesToGroups($data = []) {
        $data = (array) $data;
        $this->db->query(
            "UPDATE ".$this->db->table('customer_preferences_to_groups')."
            SET
               `editable_by_customer` = ".(int) $data['editable'].",
               `visible_by_customer` = ".(int) $data['visible'].",
               `show_shipping` = '".(int) $data['show_shipping']."',
               `show_checkout_confirm` = '".(int) $data['show_checkout_confirm']."',
               `show_fast_checkout` = '".(int) $data['show_fast_checkout']."',
               `show_admin_invoice` = '".(int) $data['show_admin_invoice']."',
               `show_invoice` = '".(int) $data['show_invoice']."',
               `show_confirmation_email` = '".(int) $data['show_confirmation_email']."',
               `Status` = ".(int) $data['status']."
            WHERE
                `global_attribute_id` = ".(int) $data['attribute_id']."
                AND `customer_group_id` = ".(int) $data['customer_group_id']
        );

        $this->clearCache();
    }

    /**
     * @param int $customer_preferences_id
     * @param int $customer_group_id
     * @param array $data
     *
     * @throws AException
     */
    public function updatePreferencesValues($customer_preferences_id, $customer_group_id = 0, $data = []) {
        $language_id = $this->language->getContentLanguageID();

        if (isset($data['settings'])) {
            if (isset($this->request->get['name'])) {
                $data['name'] = $this->request->get['name'];
            }
            $this->updatePreferences($customer_preferences_id, $data);
        } else {
            $data['attribute_value_id'] = (array) $data['attribute_value_id'];
            foreach ($data['attribute_value_id'] as $key => $status) {
                $value_data = [
                    'attribute_value_id' => $data['attribute_value_id'][$key],
                    'name'               => $data['name'][$key],
                    'sort_order'         => $data['sort_order'][$key],
                    'default'            => (string)$data['default_value']===(string)$key ? 1 : 0,
                ];

                //Check if new, delete or update
                if ($status == 'delete' && strpos($key, 'new') === false) {
                    //delete this attribute value for all languages
                    $languages = $this->language->getActiveLanguages();
                    foreach ($languages as $lang) {
                        $this->language->deleteDescriptions(
                            'customer_preferences_value_descriptions_to_groups',
                            [
                                'customer_preferences_value_id' => (int) $key,
                                'customer_group_id'           => (int) $customer_group_id,
                                'language_id'                 => (int) $lang['language_id'],
                            ]
                        );
                    }
                    $sql = "DELETE FROM ".$this->db->table('customer_preferences_values_to_groups')."
                            WHERE `customer_preferences_value_id` = '".(int) $key."'
                                AND `customer_preferences_id` = '".$customer_preferences_id."'
                                AND `customer_group_id` = ".(int) $customer_group_id;
                    $this->db->query($sql);
                } else {
                    if ($status == 'new') {
                        // Need to create new attribute value
                        $sql = "INSERT INTO ".$this->db->table('customer_preferences_values_to_groups')."
                            SET `customer_preferences_id` = '".$customer_preferences_id."',
                                `sort_order` = '".(int) $value_data['sort_order']."',
                                 `default` = '".(int) $value_data['default']."',
                                `customer_group_id` = '".(int) $customer_group_id."'";
                        $this->db->query($sql);
                        $val_id = $this->db->getLastId();
                        $descr = [];
                        if ($language_id != $this->language->getDefaultLanguageID()) {
                            $all_languages = $this->language->getActiveLanguages();
                            foreach ($all_languages as $lang) {
                                $descr[$lang['language_id']] = ['value' => $value_data['name']];
                            }
                        } else {
                            $descr = [$language_id => ['value' => $value_data['name']]];
                        }
                        $this->language->replaceDescriptions(
                            'customer_preferences_value_descriptions_to_groups',
                            [
                                'customer_preferences_value_id' => (int) $val_id,
                                'customer_preferences_id'       => $customer_preferences_id,
                                'customer_group_id'           => $customer_group_id,
                            ],
                            $descr
                        );
                    } else {
                        //Existing need to update
                        $sql = "UPDATE ".$this->db->table('customer_preferences_values_to_groups')."
                            SET `sort_order` = '".(int) $value_data['sort_order']."',
                             `default` = '".(int) $value_data['default']."'
                            WHERE `customer_preferences_id` = '".$customer_preferences_id."' 
                                AND `customer_preferences_value_id` = '".$key."' 
                                    AND `customer_group_id` = ".(int) $customer_group_id;
                        $this->db->query($sql);
                        $this->language->replaceDescriptions(
                            'customer_preferences_value_descriptions_to_groups',
                            [
                                'customer_preferences_value_id' => (int) $key,
                                'customer_preferences_id'       => $customer_preferences_id,
                                'customer_group_id'           => (int) $customer_group_id,
                            ],
                            [$language_id => ['value' => $value_data['name']]]
                        );
                    }
                }
            }
        }
        $this->clearCache();
    }

    /**
     * @param int $attribute_value_id
     *
     * @throws AException
     */
    public function deleteGlobalAttributeValueDescription($attribute_value_id) {
        $this->db->query(
            "DELETE FROM ".$this->db->table('global_attributes_value_descriptions')."
            WHERE attribute_value_id = ".(int) $attribute_value_id
        );
        $this->clearCache();
    }

    /**
     * @param int $customer_id
     *
     * @return array
     * @throws AException
     */
    public function getCustomerPreferences($customer_id) {
        $sql = "SELECT customer_group_id 
                FROM ".$this->db->table('customers')." 
                WHERE customer_id = ".(int) $customer_id;
        $result = $this->db->query($sql);
        $customer_group_id = $result->row['customer_group_id'];

        $attributes = $this->db->query(
            "SELECT ca.*, cag.*
            FROM ".$this->db->table('customer_preferences')." ca
            INNER JOIN ".$this->db->table('customer_preferences_to_groups')." cag
                ON cag.customer_preferences_id = ca.customer_preferences_id
            WHERE customer_group_id = ".(int) $customer_group_id."
                AND cag.status=1
            ORDER BY ca.sort_order"
        );

        $results = $attributes->rows;
        $output = [];
        foreach ($results as $k => $item) {
            $selected_values = $this->db->query(
                "SELECT customer_preferences_value_id, customer_preferences_value
                FROM ".$this->db->table('customer_preferences_values')."
                WHERE customer_preferences_id = ".$item['customer_preferences_id']."
                    AND customer_id = '".$customer_id."'"
            );
            $item['selected_values'] = $selected_values->rows;
            $attr = $this->getPreferencesInfo($item['customer_preferences_id'], $customer_group_id);
            $output[$k] = array_merge($item, $attr[0]);
        }
        return $output;
    }

    /**
     * @param int $customer_id
     *
     * @return array
     * @throws AException
     */
    public function getCustomerPreferencesIds($customer_id) {
        $ids = [];
        $res = $this->db->query(
            "SELECT customer_preferences_id
            FROM ".$this->db->table('customer_preferences_to_groups')."
            WHERE customer_group_id = (
                SELECT customer_group_id
                FROM ".$this->db->table('customers')."
                WHERE customer_id=".(int) $customer_id." )"
        );
        foreach ($res->rows as $row) {
            $ids[] = $row['customer_preferences_id'];
        }
        return $ids;
    }

    /**
     * @param int $customer_id
     * @param array $data
     *
     * @return bool
     * @throws AException
     */
    public function updateCustomerPreferencesValues($customer_id, $data) {
        if (!has_value($data)) {
            return false;
        }

        $attributes = $this->getCustomerPreferences($customer_id);
        foreach ($attributes as $attribute) {
            $item = $data[$attribute['customer_preferences_id']];
            if (isset($data[$attribute['customer_preferences_id']])
                || (!isset($data[$attribute['customer_preferences_id']]) && $attribute['element_type'] == 'C')
            ) {
                // for unchecked checkbox
                if ((!isset($data[$attribute['customer_preferences_id']]) && $attribute['element_type'] == 'C')
                    && sizeof($data) != 1) {
                    $item = '';
                }

                $this->db->query(
                    "DELETE FROM  ".$this->db->table('customer_preferences_values')."
                    WHERE `customer_preferences_id` = '".(int) $attribute['customer_preferences_id']."'
                        AND `customer_id` = '".(int) $customer_id."'"
                );
                if (is_array($item)) {
                    foreach ($item as $val) {
                        $this->db->query(
                            "INSERT INTO ".$this->db->table('customer_preferences_values')."
                            SET `customer_id` = '".(int) $customer_id."',
                                `customer_preferences_id` = '".(int) $attribute['customer_preferences_id']."',
                                `customer_preferences_value` = '".$this->db->escape($val)."'"
                        );
                    }
                } else {
                    $this->db->query(
                        "INSERT INTO ".$this->db->table('customer_preferences_values')."
                        SET `customer_id` = '".(int) $customer_id."',
                            `customer_preferences_id` = '".(int) $attribute['customer_preferences_id']."',
                            `customer_preferences_value` = '".$this->db->escape($item)."'"
                    );
                }
            }
        }

        $this->clearCache();
        return true;
    }

    protected function clearCache() {
        $this->cache->remove('customer_preferences');
    }

    /**
     * @param $order_id
     * @param $customer_id
     * @return mixed
     * @throws AException
     */
    public function getOrderPreferences($order_id,$customer_id) {
        $query = $this->db->query("SELECT * FROM ". $this->db->table("customer_preferences_order") . " WHERE `order_id`='".$order_id."' AND `customer_id`='".$customer_id."' ORDER BY  `customer_preferences_order_id` ASC");
        return $query->rows;
    }

    /**
     * @param $customer_id
     * @return void
     * @throws AException
     */
    public function removePreferencesByCustomerId($customer_id) {
        $this->db->query("DELETE FROM ".$this->db->table("customer_preferences_values")." WHERE `customer_id`='".$customer_id."'");
    }

    /**
     * @param $customer_id
     * @return mixed
     * @throws AException
     */
    public function getCustPreferencesByCustId($customer_id) {
        $sql = "SELECT * FROM ".$this->db->table("customer_preferences_values"). " cpv 
        LEFT JOIN ".$this->db->table("customer_preferences")." cp ON cp.customer_preferences_id = cpv.customer_preferences_id 
        WHERE cpv.customer_id='".$customer_id."' ORDER BY cp.sort_order ASC";
        $query = $this->db->query($sql);
        return $query->rows;
    }

}