<?php
/*------------------------------------------------------------------------------

  For Abante Cart, E-commerce Solution
  http://www.AbanteCart.com

  Copyright (c) 2014-2024 We Hear You 2, Inc.  (WHY2)

------------------------------------------------------------------------------*/

if (!defined('DIR_CORE')) {
    header('Location: static_pages/');
}

class ModelSaleB2bCustomer extends Model {

    /**
     * @param array $data
     *
     * @return array|int
     * @throws AException
     */
    public function getTotalCustomers($data = []) {
        return $this->getCustomers($data, 'total_only');
    }

    /**
     * @param $data
     * @param $mode
     * @return int|mixed
     * @throws AException
     */
    public function getCustomers($data = [], $mode = 'default') {
        if ($mode == 'total_only' && !$this->dcrypt->active) {
            $sql = "SELECT COUNT(*) as total ";
        } else {
            $sql = "SELECT c.customer_id,
                            c.firstname,
                            c.lastname,
                            c.loginname,
                            ca.company,
                            c.email,
                            c.sms,
                            c.status,
                            c.approved,
                            c.customer_group_id,
                            c.date_added,
                            c.date_modified,
                            CONCAT(c.firstname, ' ', c.lastname) AS name,
                            cg.name AS customer_group
                            ";
        }
        if ($mode != 'total_only' && $mode != 'quick') {
            $sql .= ", (SELECT COUNT(o.order_id) as cnt
                        FROM ".$this->db->table("orders")." o
                        WHERE c.customer_id = o.customer_id AND o.order_status_id>0) as orders_count";
        }

        if ($this->dcrypt->active) {
            $sql .= ", c.key_id ";
        }

        $sql .= " FROM ".$this->db->table("customers")." c
                  LEFT JOIN ".$this->db->table("customer_groups")." cg 
                    ON (c.customer_group_id = cg.customer_group_id) 
                  LEFT JOIN ".$this->db->table("addresses")." ca ON ca.address_id = c.address_id AND ca.customer_id = c.customer_id 
                  ";

        $implode = [];
        $filter = (isset($data['filter']) ? $data['filter'] : []);

        if (isset($filter['name']) && has_value($filter['name'])) {
            $implode[] = "CONCAT(c.firstname, ' ', c.lastname) 
                LIKE '%".$this->db->escape($filter['name'], true) ."%' ";
        }
        if (isset($filter['customer_id']) && has_value($filter['customer_id'])) {
            $implode[] = "c.customer_id = ".(int) $filter['customer_id'];
        }

        if (isset($filter['name_email']) && has_value($filter['name_email'])) {
            $implode[] = "CONCAT(c.firstname, ' ', c.lastname, ' ', c.email) 
                            LIKE '%".$this->db->escape($filter['name_email'], true )."%' ";
        }

        if (isset($filter['company']) && has_value($filter['company'])) {
            $implode[] = "ca.company LIKE '%".$this->db->escape($filter['company'], true )."%' ";
        }

        //more specific login, last and first name search
        if (isset($filter['loginname']) && has_value($filter['loginname'])) {
            $implode[] =
                "LOWER(c.loginname) = LOWER('".$this->db->escape($filter['loginname'])."') ";
        }
        if (isset($filter['firstname']) && has_value($filter['firstname'])) {
            $implode[] = "LOWER(c.firstname) LIKE LOWER('".$this->db->escape($filter['firstname'], true) ."%') ";
        }
        if (isset($filter['lastname']) && has_value($filter['lastname'])) {
            $implode[] = "LOWER(c.lastname) LIKE LOWER('".$this->db->escape($filter['lastname'], true) ."%') ";
        }
        //select differently if encrypted
        if (!$this->dcrypt->active) {
            if (isset($filter['email']) && has_value($filter['email'])) {
                $implode[] = "c.email LIKE '%".$this->db->escape($filter['email'], true)."%' ";
            }
            if (isset($filter['telephone']) && has_value($filter['telephone'])) {
                $implode[] = "c.telephone LIKE '%".$this->db->escape($filter['telephone'], true)."%' ";
            }
            if (isset($filter['sms']) && has_value($filter['sms'])) {
                $implode[] = "c.sms LIKE '%".$this->db->escape($filter['sms'], true)."%' ";
            }
        }

        if (isset($filter['customer_group_id']) && has_value($filter['customer_group_id'])) {
            $implode[] = "cg.customer_group_id = '".$this->db->escape($filter['customer_group_id'])."'";
        }
        // select only subscribers (group + customers with subscription)
        if (isset($filter['all_subscribers']) && has_value($filter['all_subscribers'])) {
            $implode[] = "( (c.newsletter=1 AND c.status = 1 AND c.approved = 1) 
                OR
                (c.newsletter=1 AND cg.customer_group_id = '".(int) $this->getSubscribersCustomerGroupId()."'))";
        }

        // select only customers without newsletter subscribers
        if (isset($filter['only_customers']) && has_value($filter['only_customers'])) {
            $implode[] = "cg.customer_group_id NOT IN (".(int) $this->getSubscribersCustomerGroupId().") ";
        }

        if (isset($filter['only_with_mobile_phones']) && has_value($filter['only_with_mobile_phones'])) {
            $implode[] = " TRIM(COALESCE(c.sms,''))  <> '' ";
        }

        if (isset($filter['status']) && has_value($filter['status'])) {
            $implode[] = "c.status = '".(int) $filter['status']."'";
        }

        if (isset($filter['approved']) && has_value($filter['approved'])) {
            $implode[] = "c.approved = '".(int) $filter['approved']."'";
        }

        if (isset($filter['date_added']) && has_value($filter['date_added'])) {
            $implode[] = "DATE(c.date_added) = DATE('".$this->db->escape($filter['date_added'])."')";
        }

        $store_id = null;
        if (isset($this->session->data['current_store_id']) && has_value($this->session->data['current_store_id'])) {
            $store_id = (int) $this->session->data['current_store_id'];
        }

        $this->load->model('setting/store');
        if (!$store_id && !$this->model_setting_store->isDefaultStore()) {
            $store_id = $this->config->get('current_store_id');
        }

        if ($store_id !== null) {
            $implode[] = "c.store_id = ".(int) $store_id;
        }

        if (
            (
                ($filter['all_subscribers'] ?? null) || ($filter['only_subscribers'] ?? null)
            )
            && $filter['newsletter_protocol']
        ) {
            $sql .= "RIGHT JOIN ".$this->db->table('customer_notifications')." cn
                    ON (cn.customer_id = c.customer_id
                        AND cn.sendpoint='newsletter'
                        AND cn.status=1
                        AND cn.protocol = '".$this->db->escape($filter['newsletter_protocol'])."') ";
        }

        if ($implode) {
            $sql .= " WHERE ".implode(" AND ", $implode);
        }

        //If for total, we done building the query
        if ($mode == 'total_only' && !$this->dcrypt->active) {
            $query = $this->db->query($sql);
            return $query->row['total'];
        }

        $sort_data = [
            'customer_id'    => 'c.customer_id',
            'name'           => 'name',
            'loginname'      => 'c.loginname',
            'lastname'       => 'c.lastname',
            'company'        => 'ca.company',
            'email'          => 'c.email',
            'sms'            => 'c.sms',
            'customer_group' => 'customer_group',
            'status'         => 'c.status',
            'approved'       => 'c.approved',
            'date_added'     => 'c.date_added',
            'orders_count'   => 'orders_count',
        ];

        //Total calculation for encrypted mode
        // NOTE: Performance slowdown might be noticed or larger search results
        if ($mode != 'total_only') {
            $sql .= " ORDER BY ".($sort_data[$data['sort']] ? $sort_data[$data['sort']] : 'name');
            if (isset($data['order']) && (strtoupper($data['order']) == 'DESC')) {
                $sql .= " DESC";
            } else {
                $sql .= " ASC";
            }

            if (isset($data['start']) || isset($data['limit'])) {
                if ($data['start'] < 0) {
                    $data['start'] = 0;
                }
                if ($data['limit'] < 1) {
                    $data['limit'] = 20;
                }
                $sql .= " LIMIT ".(int) $data['start'].",".(int) $data['limit'];
            }
        }

        $query = $this->db->query($sql);
        $result_rows = $query->rows;
        if ($this->dcrypt->active) {
            if (isset($filter['email']) && has_value($filter['email'])) {
                $result_rows = $this->_filter_by_encrypted_field($result_rows, 'email', $filter['email']);
            }
            if (isset($filter['telephone']) && has_value($filter['telephone'])) {
                $result_rows = $this->_filter_by_encrypted_field($result_rows, 'telephone', $filter['telephone']);
            }
            if (isset($filter['sms']) && has_value($filter['sms'])) {
                $result_rows = $this->_filter_by_encrypted_field($result_rows, 'sms', $filter['sms']);
            }
        }

        if ($mode == 'total_only') {
            //we get here only if in data encryption mode
            return count($result_rows);
        }
        //finally decrypt data and return result
        for ($i = 0; $i < count($result_rows); $i++) {
            $result_rows[$i] = $this->dcrypt->decrypt_data($result_rows[$i], 'customers');
        }

        return $result_rows;
    }

    /**
     * @return mixed
     * @throws AException
     */
    public function getCompanies() {
        $sql = "SELECT DISTINCT ca.company FROM ".$this->db->table("addresses")." ca 
               LEFT JOIN ".$this->db->table("customers")." c ON c.customer_id = ca.customer_id AND ca.address_id = c.address_id";
        $query = $this->db->query($sql);
        return $query->rows;
    }

    /**
     * @param array $data
     * @param string $field
     * @param mixed $value
     *
     * @return array
     * @throws AException
     */
    private function _filter_by_encrypted_field($data, $field, $value) {
        if (!count($data)) {
            return [];
        }
        if (!has_value($field) || !has_value($value)) {
            return $data;
        }
        $result_rows = [];
        foreach ($data as $result) {
            if ($this->dcrypt->active) {
                $f_value = $this->dcrypt->decrypt_field($result[$field], $result['key_id']);
            } else {
                $f_value = $result[$field];
            }
            if (!(strpos(strtolower($f_value), strtolower($value)) === false)) {
                $result_rows[] = $result;
            }
        }
        return $result_rows;
    }

    /**
     * @return int
     * @throws AException
     */
    public function getSubscribersCustomerGroupId()
    {
        $query = $this->db->query(
            "SELECT customer_group_id
                                    FROM `".$this->db->table("customer_groups")."`
                                    WHERE `name` = 'Newsletter Subscribers'
                                    LIMIT 0,1"
        );
        return !$query->row['customer_group_id'] ? (int) $this->config->get('config_customer_group_id')
            : (int) $query->row['customer_group_id'];
    }

}