<?php
/*------------------------------------------------------------------------------

  For Abante Cart, E-commerce Solution
  http://www.AbanteCart.com

  Copyright (c) 2014-2024 We Hear You 2, Inc.  (WHY2)

------------------------------------------------------------------------------*/

if (!defined('DIR_CORE')) {
    header('Location: static_pages/');
}

/**
 * ControllerResponsesListingGridB2bOrder
 * @property ModelSaleB2bOrder $model_sale_b2b_order
 * @property ModelExtensionCompanyReferenceLicenseInfo $model_extension_company_reference_license_info
 */
class ControllerResponsesListingGridB2bOrder extends AController {
    public $error = [];
    public $data = [];

    public function verifyCompanyRef() {
        if ($this->config->get('company_reference_status')) {
            $this->load->model('extension/company_reference_license_info');
            $license_key = $this->config->get('company_reference_license_code');
            if ($license_key) {
                $exist = $this->db->query("SHOW TABLES LIKE '" . $this->db->table('company_reference_license_info') . "'");
                if ($exist->num_rows === 1) {
                    $store_id = $this->model_extension_company_reference_license_info->getStoreId();
                    $license_key = $this->model_extension_company_reference_license_info->getLicenseCode($store_id);
                    $license_info = $this->model_extension_company_reference_license_info->getLicenseData($store_id);
                    $path_to_phpseclib = DIR_SYSTEM . "lib/vendor/phpseclib/phpseclib/phpseclib/";
                    $license = new License($path_to_phpseclib);
                    $license_manager = $license->getManager();
                    $license_manager->setKeys($license_key, $license_info['public_key'], 'company_reference');
                    $ttl = 1209600;
                    $validate = $license_manager->validate($license_info['license_data'], $ttl);
                    $store_status = $this->model_extension_company_reference_license_info->getLicensedStoreStatus();
                    if ($validate['status'] === 'valid' && $store_status == 1) {
                        return true;
                    } else {
                        return false;
                    }
                }
            }
        } else {
            return false;
        }
    }

    public function main() {
        //init controller data
        $this->extensions->hk_InitData($this, __FUNCTION__);

        $this->loadLanguage('sale/order');
        $this->loadModel('sale/order');
        $this->loadModel('sale/b2b_order');

        $page = $this->request->post['page']; // get the requested page
        $limit = $this->request->post['rows']; // get how many rows we want to have into the grid
        $sidx = $this->request->post['sidx']; // get index row - i.e. user click to sort
        $sord = $this->request->post['sord']; // get the direction

        // process jGrid search parameter
        $allowedFields = array_merge(
            [
                'name',
                'order_id',
                'date_added',
                'reference_number',
                'payment_company',
                'total'
            ],
            (array) $this->data['allowed_fields']
        );
        $allowedSortFields = array_merge(
            [
                'customer_id',
                'order_id',
                'name',
                'payment_company',
                'reference_number',
                'status',
                'date_added',
                'total'
            ],
            (array) $this->data['allowed_sort_fields']
        );

        $allowedDirection = ['asc', 'desc'];

        if (!in_array($sidx, $allowedSortFields)) {
            $sidx = $allowedSortFields[0];
        }
        if (!in_array($sord, $allowedDirection)) {
            $sord = $allowedDirection[0];
        }

        if (in_array($sidx, ['customer_id', 'order_id','payment_company', 'date_added', 'total'])) {
            $sidx = 'o.'.$sidx;
        } elseif (in_array($sidx, ['reference_number'])) {
            $sidx = 'ref.' . $sidx;
        }

        $data = [
            'sort'  => $sidx,
            'order' => $sord,
            'start' => ($page - 1) * $limit,
            'limit' => $limit,
        ];

        $data['company_ref_status'] = $this->verifyCompanyRef();

        if (isset($this->request->get['store_id'])) {
            $data['store_id'] = $this->request->get['store_id'];
            $data['store_id'] = $data['store_id'] != 'all' ? (int) $data['store_id'] : 'all';
        }

        if (isset($this->request->get['status']) && $this->request->get['status'] != '') {
            $data['filter_order_status_id'] = $this->request->get['status'];
        }
        if (has_value($this->request->get['customer_id'])) {
            $data['filter_customer_id'] = $this->request->get['customer_id'];
        }
        if (has_value($this->request->get['product_id'])) {
            $data['filter_product_id'] = $this->request->get['product_id'];
        }
        if (has_value($this->request->get['reference_number'])) {
            $data['filter_reference_number'] = $this->request->get['reference_number'];
        }
        if (has_value($this->request->get['payment_company'])) {
            $data['filter_payment_company'] = $this->request->get['payment_company'];
        }

        if (isset($this->request->post['_search']) && $this->request->post['_search'] == 'true') {
            $searchData = json_decode(htmlspecialchars_decode($this->request->post['filters']), true);
            if ($searchData['rules']) {
                foreach ($searchData['rules'] as $rule) {
                    if (!in_array($rule['field'], $allowedFields)) {
                        continue;
                    }
                    if ($rule['field']==='payment_company') {
                        $data['filter_sub_payment_company'] = trim($rule['data']);
                    } elseif ($rule['field']==='reference_number') {
                        $data['filter_sub_reference_number'] = trim($rule['data']);
                    } else {
                        $data['filter_'.$rule['field']] = trim($rule['data']);
                    }
                    if ($rule['field'] == 'date_added') {
                        $data['filter_'.$rule['field']] = dateDisplay2ISO($rule['data']);
                    }
                }
            }
        }

        $this->loadModel('localisation/order_status');
        $results = $this->model_localisation_order_status->getOrderStatuses();
        $statuses = ['' => $this->language->get('text_select_status'),];
        foreach ($results as $item) {
            $statuses[$item['order_status_id']] = $item['name'];
        }

        $total = $this->model_sale_b2b_order->getTotalOrders($data);
        if ($total > 0) {
            $total_pages = ceil($total / $limit);
        } else {
            $total_pages = 0;
        }

        if ($page > $total_pages) {
            $page = $total_pages;
            $data['start'] = ($page - 1) * $limit;
        }

        $response = new stdClass();
        $response->page = $page;
        $response->total = $total_pages;
        $response->records = $total;

        $results = $this->model_sale_b2b_order->getOrders($data);

        $i = 0;
        if ($this->verifyCompanyRef()) {
            foreach ($results as $result) {
                $response->rows[$i]['id'] = $result['order_id'];
                $response->rows[$i]['cell'] = [
                    $result['order_id'],
                    $result['reference_number'],
                    $result['name'],
                    $result['payment_company'],
                    $this->html->buildSelectBox(
                        [
                            'name' => 'order_status_id[' . $result['order_id'] . ']',
                            'value' => array_search($result['status'], $statuses),
                            'options' => $statuses,
                        ]
                    ),
                    dateISO2Display($result['date_added'], $this->language->get('date_format_short')),
                    $this->currency->format($result['total'], $result['currency'], $result['value']),
                ];
                $i++;
            }
        } else {
            foreach ($results as $result) {
                $response->rows[$i]['id'] = $result['order_id'];
                $response->rows[$i]['cell'] = [
                    $result['order_id'],
                    $result['name'],
                    $result['payment_company'],
                    $this->html->buildSelectBox(
                        [
                            'name' => 'order_status_id[' . $result['order_id'] . ']',
                            'value' => array_search($result['status'], $statuses),
                            'options' => $statuses,
                        ]
                    ),
                    dateISO2Display($result['date_added'], $this->language->get('date_format_short')),
                    $this->currency->format($result['total'], $result['currency'], $result['value']),
                ];
                $i++;
            }
        }
        $this->data['response'] = $response;
        //update controller data
        $this->extensions->hk_UpdateData($this, __FUNCTION__);
        $this->load->library('json');
        $this->response->setOutput(AJson::encode($this->data['response']));
    }

    public function update()
    {
        //init controller data
        $this->extensions->hk_InitData($this, __FUNCTION__);

        $this->loadModel('sale/order');
        $this->loadLanguage('sale/order');
        if (!$this->user->canModify('listing_grid/b2b_order')) {
            $error = new AError('');
            return $error->toJSONResponse(
                'NO_PERMISSIONS_402',
                [
                    'error_text'  => sprintf($this->language->get('error_permission_modify'), 'listing_grid/b2b_order'),
                    'reset_value' => true,
                ]
            );
        }

        switch ($this->request->post['oper']) {
            case 'del':
                $ids = explode(',', $this->request->post['id']);
                if (!empty($ids)) {
                    foreach ($ids as $id) {
                        $this->model_sale_order->deleteOrder($id);
                    }
                }
                break;
            case 'save':
                $ids = explode(',', $this->request->post['id']);
                if (!empty($ids)) {
                    foreach ($ids as $id) {
                        $this->model_sale_order->editOrder(
                            $id, ['order_status_id' => $this->request->post['order_status_id'][$id]]
                        );
                    }
                }
                break;
            default:
        }

        //update controller data
        $this->extensions->hk_UpdateData($this, __FUNCTION__);
        return null;
    }

    /**
     * update only one field
     *
     * @return void
     * @throws AException
     */
    public function update_field() {
        //init controller data
        $this->extensions->hk_InitData($this, __FUNCTION__);

        $this->loadLanguage('sale/order');
        $this->loadModel('sale/order');

        if (!$this->user->canModify('listing_grid/b2b_order')) {
            $error = new AError('');
            return $error->toJSONResponse(
                'NO_PERMISSIONS_402',
                [
                    'error_text'  => sprintf(
                        $this->language->get('error_permission_modify'),
                        'listing_grid/b2b_order'
                    ),
                    'reset_value' => true,
                ]
            );
        }

        if (has_value($this->request->post['downloads'])) {
            $data = $this->request->post['downloads'];
            $this->loadModel('catalog/download');
            foreach ($data as $order_download_id => $item) {
                if (isset($item['expire_date'])) {
                    $item['expire_date'] = $item['expire_date']
                        ? dateDisplay2ISO($item['expire_date'], $this->language->get('date_format_short'))
                        : '';
                }
                $this->model_catalog_download->editOrderDownload($order_download_id, $item);
            }
            return null;
        }

        if (isset($this->request->get['id'])) {
            $this->model_sale_order->editOrder($this->request->get['id'], $this->request->post);
            return null;
        }

        //request sent from jGrid. ID is key of array
        foreach ($this->request->post as $field => $value) {
            foreach ($value as $k => $v) {
                $this->model_sale_order->editOrder($k, [$field => $v]);
            }
        }

        //update controller data
        $this->extensions->hk_UpdateData($this, __FUNCTION__);
    }

    public function summary()
    {
        //update controller data
        $this->extensions->hk_InitData($this, __FUNCTION__);

        $this->loadLanguage('sale/order');
        $this->loadModel('sale/order');

        $response = new stdClass();

        if (isset($this->request->get['order_id'])) {
            $order_id = $this->request->get['order_id'];
        } else {
            $order_id = 0;
        }

        $order_info = $this->model_sale_order->getOrder($order_id);

        if (empty($order_info)) {
            $response->error = $this->language->get('error_order_load');
        } else {
            $response->order = [
                'order_id'        => '#'.$order_info['order_id'],
                'name'            => $order_info['firstname'].''.$order_info['lastname'],
                'email'           => $order_info['email'],
                'telephone'       => $order_info['telephone'],
                'date_added'      => dateISO2Display(
                                                        $order_info['date_added'],
                                                        $this->language->get('date_format_short')
                                                    ),
                'total'           => $this->currency->format(
                    $order_info['total'], $order_info['currency'], $order_info['value']
                ),
                'order_status'    => $order_info['order_status_id'],
                'shipping_method' => $order_info['shipping_method'],
                'payment_method'  => $order_info['payment_method'],
            ];

            if ($order_info['customer_id']) {
                $response->order['name'] = '<a href="'
                    .$this->html->getSecureURL(
                        'sale/customer/update',
                        '&customer_id='.$order_info['customer_id'])
                    .'">'
                    .$response->order['name'].'</a>';
            }

            $this->loadModel('localisation/order_status');
            $status = $this->model_localisation_order_status->getOrderStatus($order_info['order_status_id']);
            if ($status) {
                $response->order['order_status'] = $status['name'];
            }
        }
        $this->data['response'] = $response;
        //update controller data
        $this->extensions->hk_UpdateData($this, __FUNCTION__);

        $this->load->library('json');
        $this->response->setOutput(AJson::encode($this->data['response']));
    }

}