<?php
/*------------------------------------------------------------------------------

  For Abante Cart, E-commerce Solution
  http://www.AbanteCart.com

  Copyright (c) 2014-2024 We Hear You 2, Inc.  (WHY2)

------------------------------------------------------------------------------*/
if (!defined('DIR_CORE')){
    header('Location: static_pages/');
}

/**
 * Class AHcaptchaHtml
 * @method SelectboxHtmlElement buildSelectBox(array $data)
 * @method MultiSelectboxHtmlElement buildMultiSelectBox(array $data)
 * @method HiddenHtmlElement buildHidden(array $data)
 * @method MultivalueHtmlElement buildMultivalue(array $data)
 * @method MultivalueListHtmlElement buildMultivalueList(array $data)
 * @method SubmitHtmlElement buildSubmit(array $data)
 * @method InputHtmlElement buildInput(array $data)
 * @method PasswordHtmlElement buildPassword(array $data)
 * @method TextAreaHtmlElement buildTextarea(array $data)
 * @method TextEditorHtmlElement buildTextEditor(array $data)
 * @method CheckboxHtmlElement buildCheckbox(array $data)
 * @method CheckboxGroupHtmlElement buildCheckboxGroup(array $data)
 * @method FileHtmlElement buildFile(array $data)
 * @method RadioHtmlElement buildRadio(array $data)
 * @method ButtonHtmlElement buildButton(array $data)
 * @method RatingHtmlElement buildRating(array $data)
 * @method CaptchaHtmlElement buildCaptcha(array $data)
 * @method ReCaptchaHtmlElement buildReCaptcha(array $data)
 * @method PasswordSetHtmlElement buildPasswordSet(array $data)
 * @method ResourceHtmlElement buildResource(array $data)
 * @method ResourceImageHtmlElement buildResourceImage(array $data)
 * @method DateHtmlElement buildDate(array $data)
 * @method EmailHtmlElement buildEmail(array $data)
 * @method NumberHtmlElement buildNumber(array $data)
 * @method PhoneHtmlElement buildPhone(array $data)
 * @method IPaddressHtmlElement buildIPaddress(array $data)
 * @method CountriesHtmlElement buildCountries(array $data)
 * @method ZonesHtmlElement buildZones(array $data)
 * @method PaginationHtmlElement buildPagination(array $data)
 * @method ModalHtmlElement buildModal(array $data)
 * @method LabelHtmlElement buildLabel(array $data)
 * @method HcaptchaHtmlElement buildHcaptcha(array $data)
 */

class AHcaptchaHtml extends AHtml {
    /**
     * @var Registry
     */

    protected $registry;
    /**
     * @var array
     */
    protected $args = [];
    /**
     * @var ARequest
     */
    protected $request;

    /**
     * @param Registry $registry
     * @param array $args
     */
    public function __construct($registry, $args = []){
        $this->registry = $registry;
        $this->request = $this->registry->get('request');
    }

    /**
     * Magic method for html-element classes calls
     *
     * @param $function_name
     * @param $args
     *
     * @return null|string
     * @throws AException
     */
    public function __call($function_name, $args){
        $class_name = ltrim($function_name, 'build').'HtmlElement';
        if (class_exists($class_name)) {
            /**
             * @var SelectboxHtmlElement|HiddenHtmlElement| $item
             */
            $item = new $class_name($args[0]);
            if (method_exists($item, 'getHtml')) {
                return $item->getHtml();
            } else {
                unset($item);
            }
        }
        return null;
    }

    /**
     * PR Build sub URL
     *
     * @param string $rt
     * @param string $params
     *
     * @return string
     * @throws AException
     */
    private function buildURL($rt, $params = ''){
        $suburl = '';
        //#PR Add admin path if we are in admin
        if (IS_ADMIN){
            $suburl .= '&s=' . ADMIN_PATH;
        }
        //add template if present
        if (!empty($this->request->get['sf'])){
            $suburl .= '&sf=' . $this->request->get['sf'];
        }

        //if in embed mode add response prefix
        if ($this->registry->get('config')->get('embed_mode') == true){
            $suburl .= '&embed_mode=1';
            if(substr($rt, 0, 2) != 'r/'){
                $rt = 'r/'.$rt;
            }
        }

        $suburl = '?' . ($rt ? 'rt=' . $rt : '') . $params . $suburl;
        return $suburl;
    }

    /**
     * Get non-secure home URL.
     *
     * @return string
     *
     * Note: Non-secure URL is base on store_url setting. If this setting is using https URL, all URLs will be secure
     * @throws AException
     */
    public function getHomeURL()
    {
        $seo_prefix = $this->registry->get('config')->get('seo_prefix');

        //for embed mode get home link with getURL
        if ($this->registry->get('config')->get('embed_mode')) {
            return $this->getURL('index/home');
        } else {
            //get config_url first
            $home_url = $this->registry->get('config')->get('config_url').$seo_prefix;
            if (!$home_url) {
                $home_url = defined('HTTP_SERVER')
                    ? HTTP_SERVER.$seo_prefix
                    : 'http://'.REAL_HOST.get_url_path($_SERVER['PHP_SELF']);
            }
            return $home_url;
        }
    }

    /**
     * Get non-secure URL. Read note below.
     *
     * @param string $rt
     * @param string $params
     * @param string $encode
     *
     * @return string
     *
     * Note: Non-secure URL is base on store_url setting. If this setting is using https URL, all URLs will be secure
     * @throws AException
     */
    public function getNonSecureURL($rt, $params = '', $encode = ''){
        return $this->getURL($rt, $params, $encode, true);
    }

    /**
     * Get URL with auto-detection for protocol
     *
     * @param string $rt
     * @param string $params
     * @param string $encode
     * @param bool $nonsecure - force to be non-secure
     *
     * @return string
     * @throws AException
     */
    public function getURL($rt, $params = '', $encode = '', $nonsecure = false)
    {
        $seo_prefix = $this->registry->get('config')->get('seo_prefix');

        //detect if request is using HTTPS
        if ($nonsecure === false && HTTPS === true) {
            $server = defined('HTTPS_SERVER')
                ? HTTPS_SERVER.$seo_prefix
                : 'https://'.REAL_HOST.get_url_path($_SERVER['PHP_SELF']);
        } else {
            //to prevent garbage session need to check constant HTTP_SERVER
            $server = defined('HTTP_SERVER')
                ? HTTP_SERVER.$seo_prefix
                : 'http://'.REAL_HOST.get_url_path($_SERVER['PHP_SELF']);
        }

        if ($this->registry->get('config')->get('storefront_template_debug')
            && isset($this->request->get['tmpl_debug'])
        ) {
            $params .= '&tmpl_debug='.$this->request->get['tmpl_debug'];
        }
        // add session id for cross-domain transition in secure mode
        if ($this->registry->get('config')->get('config_shared_session') && HTTPS === true) {
            $params .= '&session_id='.session_id();
        }

        //add token for embed mode with forbidden 3d-party cookies
        if ($this->registry->get('session')->data['session_mode'] == 'embed_token') {
            $params .= '&'.EMBED_TOKEN_NAME.'='.session_id();
        }
        return $server.INDEX_FILE.$this->url_encode($this->buildURL($rt, $params), $encode);
    }

    /**
     * Build secure URL with session token
     *
     * @param string $rt
     * @param string $params
     * @param string $encode
     *
     * @return string
     * @throws AException
     */
    public function getSecureURL($rt, $params = '', $encode = '')
    {
        $session = $this->registry->get('session');
        $config = $this->registry->get('config');
        $seo_prefix = $config->get('seo_prefix');
        // add session id for cross-domain transition in non-secure mode
        if ($config->get('config_shared_session') && HTTPS !== true) {
            $params .= '&session_id='.session_id();
        }

        $subUrl = $this->buildURL($rt, $params);

        if (IS_ADMIN === true || (defined('IS_API') && IS_API === true)) {
            //Add session token for admin and API
            if (isset($session->data['token']) && $session->data['token']) {
                $subUrl .= '&token='.$this->registry->get('session')->data['token'];
            }
        }

        //add token for embed mode with forbidden 3d-party cookies
        if ($session->data['session_mode'] == 'embed_token') {
            $subUrl .= '&'.EMBED_TOKEN_NAME.'='.session_id();
        }

        if ($config->get('storefront_template_debug') && isset($this->request->get['tmpl_debug'])) {
            $subUrl .= '&tmpl_debug='.$this->request->get['tmpl_debug'];
        }

        return HTTPS_SERVER.$seo_prefix.INDEX_FILE.$this->url_encode($subUrl, $encode);
    }

    /**
     * Build non-secure SEO URL
     *
     * @param string $rt
     * @param string $params
     * @param string $encode
     *
     * @return string
     * @throws AException
     */
    public function getSEOURL($rt, $params = '', $encode = ''){
        //skip SEO for embed mode
        if ($this->registry->get('config')->get('embed_mode')){
            return $this->getURL($rt, $params);
        }
        $this->loadModel('tool/seo_url');
        //#PR Generate SEO URL based on standard URL
        //NOTE: SEO URL is non-secure url
        return $this->url_encode($this->model_tool_seo_url->rewrite($this->getNonSecureURL($rt, $params)), $encode);
    }

    /**
     * Build secure SEO URL
     *
     * @param string $rt
     * @param string $params
     * @param string $encode
     *
     * @return string
     * @throws AException
     */
    public function getSecureSEOURL($rt, $params = '', $encode = ''){
        //add token for embed mode with forbidden 3d-party cookies
        if ($this->registry->get('session')->data['session_mode'] == 'embed_token') {
            $params .= '&'.EMBED_TOKEN_NAME.'='.session_id();
        }
        //#PR Generate SEO URL based on standard URL
        $this->loadModel('tool/seo_url');
        return $this->url_encode($this->model_tool_seo_url->rewrite($this->getSecureURL($rt, $params)), $encode);
    }

    /**
     * This builds URL to the catalog to be used in admin
     *
     * @param string $rt
     * @param string $params
     * @param string $encode
     * @param bool $ssl
     *
     * @return string
     * @throws AException
     */
    public function getCatalogURL($rt, $params = '', $encode = '', $ssl = false)
    {
        $seo_prefix = $this->registry->get('config')->get('seo_prefix');
        //add token for embed mode with forbidden 3d-party cookies
        if ($this->registry->get('session')->data['session_mode'] == 'embed_token') {
            $params .= '&'.EMBED_TOKEN_NAME.'='.session_id();
        }
        $subUrl = '?'.($rt ? 'rt='.$rt : '').$params;

        if ($this->registry->get('config')->get('config_ssl') == 2) {
            $ssl = true;
        }

        if ($ssl && parse_url($this->registry->get('config')->get('config_ssl_url'), PHP_URL_SCHEME) == 'https') {
            $HTTPS_SERVER = $this->registry->get('config')->get('config_ssl_url').$seo_prefix;
        } else {
            $HTTPS_SERVER = HTTPS_SERVER.$seo_prefix;
        }
        $http = $ssl ? $HTTPS_SERVER : HTTP_SERVER.$seo_prefix;
        return $http.INDEX_FILE.$this->url_encode($subUrl, $encode);
    }

    /**
     * encode URL for & to be &amp
     *
     * @param string $url
     * @param bool $encode
     *
     * @return string
     */
    public function url_encode($url, $encode = false)
    {
        if ($encode) {
            return str_replace('&', '&amp;', $url);
        } else {
            return $url;
        }
    }

    /**
     * Current URL built based on get params with ability to exclude params
     *
     * @param $filter_params array - array of vars to filter
     *
     * @return string - url without unwanted filter parameters
     * @throws AException
     */
    public function currentURL($filter_params = [])
    {
        $params_arr = $this->request->get;
        //detect if there is RT in the params.
        $rt = 'index/home';
        if (has_value($params_arr['rt'])) {
            $rt = $params_arr['rt'];
            $filter_params[] = 'rt';
        }
        if (has_value($params_arr['s'])) {
            $filter_params[] = 's';
        }
        $URI = '&'.$this->buildURI($params_arr, $filter_params);
        return $this->getURL($rt, $URI);
    }

    /**
     * URI encrypt parameters in URI
     *
     * @param $uri
     *
     * @return string - url without unwanted filter parameters
     * @internal param array $filter_params - array of vars to filter
     */
    public function encryptURI($uri)
    {
        $encrypted = base64_encode($uri);
        if (strlen($encrypted) <= 250) {
            return '__e='.$encrypted;
        } else {
            return $uri;
        }
    }

    /**
     * Build URI from array provided
     *
     * @param $params_arr    array - data array to process
     * @param $filter_params array - array of vars to filter
     *
     * @return string - url without unwanted filter parameters
     */
    public function buildURI($params_arr, $filter_params = [])
    {
        foreach ($filter_params as $rv) {
            unset($params_arr[$rv]);
        }

        return urldecode(http_build_query($params_arr, '', '&'));
    }

    /**
     * Filter query parameters from url.
     *
     * @param $url           string - url to process
     * @param $filter_params string|array - single var or array of vars
     *
     * @return string - url without unwanted filter query parameters
     */
    public function filterQueryParams($url, $filter_params = [])
    {
        list($url_part, $q_part) = explode('?', $url);
        parse_str($q_part, $q_vars);
        //build array if passed as string
        if (!is_array($filter_params)) {
            $filter_params = [$filter_params];
        }
        foreach ($filter_params as $rv) {
            unset($q_vars[$rv]);
        }
        foreach ($q_vars as $key => $val) {
            $q_vars[$key] = $this->request->clean($val);
        }

        $new_qs = urldecode(http_build_query($q_vars, '', '&'));
        return $url_part.'?'.$new_qs;
    }

    /**
     * remove get parameters from url.
     *
     * @param $url - url to process
     * @param $remove_vars
     *
     * @return string - url without unwanted get parameters
     * @internal   param array|string $vars - single var or array of vars
     * @deprecated since 1.1.4! Use filterQueryParams() instead
     *
     */
    public function removeQueryVar($url, $remove_vars)
    {
        return $this->filterQueryParams($url, $remove_vars);
    }

    /**
     * function returns text error or empty
     *
     * @param string $query
     * @param string $keyword
     *
     * @return string
     * @throws AException
     */
    public function isSEOKeywordExists($query, $keyword = '')
    {
        if (!$keyword) {
            return '';
        }
        $seo_key = SEOEncode($keyword);

        $db = $this->registry->get('db');
        $sql = "SELECT *
				FROM ".$db->table('url_aliases')."
				WHERE query<>'".$db->escape($query)."' AND keyword='".$db->escape($seo_key)."'";
        $result = $db->query($sql);
        if ($result->num_rows) {
            $url = HTTP_CATALOG.$seo_key;
            return sprintf($this->language->get('error_seo_keyword'), $url, $seo_key);
        }

        return '';
    }

    /**
     * create html code based on passed data
     *
     * @param  $data - array with element data
     *               sample
     *               $data = array(
     *               'type' => 'input' //(hidden, textarea, selectbox, file...)
     *               'name' => 'input name'
     *               'value' => 'input value' // could be array for select
     *               'style' => 'my-form'
     *               'form' => 'form id' // needed for unique element ID     *
     *               );
     *
     * @return object
     * @throws AException
     */
    public function builHcaptchaElement($data)
    {
        return HtmlHcaptchaElementFactory::create($data);
    }

    /**
     * @return string
     * @throws AException
     */
    public function getStoreSwitcher(){
        $registry = $this->registry;
        $view = new AView($this->registry, 0);
        //check if store_id is passed or in the session
        $store_id = $registry->get('config')->get('current_store_id');
        //set store selector
        $stores = [];
        $hidden = [];
        $stores[0] = ['name' => $registry->get('language')->get('text_default')];
        $registry->get('load')->model('setting/store');
        /**
         * @var ModelSettingStore $model
         */
        $model = $registry->get('model_setting_store');
        //if loaded not default store - hide store switcher
        $default_store_settings = $model->getStore(0);
        if ($this->registry->get('config')->get('config_url') != $default_store_settings['config_url']) {
            return '';
        }
        $result_stores = $model->getStores();
        if (sizeof($result_stores) > 0) {
            foreach ($result_stores as $rs) {
                $stores[$rs['store_id']] = [
                    'name'     => $rs['alias'] ? : $rs['name'],
                    'store_id' => $rs['store_id'],
                ];
            }
            foreach ($registry->get('request')->get as $name => $value) {
                if ($name == 'store_id') {
                    continue;
                }
                $hidden[$name] = $value;
            }
            $view->assign('all_stores', $stores);
            $view->assign('current_store', $stores[$store_id]['name']);
            $view->assign('hiddens', $hidden);
            $view->assign('text_select_store', $registry->get('language')->get('text_select_store'));
            return $view->fetch('form/store_switcher.tpl');
        } else {
            return '';
        }
    }

    /**
     * @return string
     * @throws AException
     */
    public function getContentLanguageSwitcher(){
        $registry = $this->registry;
        $view = new AView($this->registry, 0);
        $registry->get('load')->model('localisation/language');
        $results = $registry->get('model_localisation_language')->getLanguages();
        $template['languages'] = [];

        foreach ($results as $result){
            if ($result['status']){
                $template['languages'][] = [
                    'name' => $result['name'],
                    'code' => $result['code'],
                    'image' => $result['image'],
                ];
            }
        }
        if (sizeof($template['languages']) > 1){
            //selected in selectbox
            $template['language_code'] = $registry->get('language')->getContentLanguageCode();
            foreach ($registry->get('request')->get as $name => $value){
                if ($name == 'content_language_code') {
                    continue;
                }
                $template['hiddens'][$name] = $value;
            }
        } else{
            $template['languages'] = [];
        }
        $view->batchAssign($template);
        return $view->fetch('form/language_switcher.tpl');
    }

    /**
     * @return string
     * @throws AException
     */
    public function getContentLanguageFlags(){
        $registry = $this->registry;
        $view = new AView($this->registry, 0);
        $registry->get('load')->model('localisation/language');
        $results = $registry->get('model_localisation_language')->getLanguages();
        $template['languages'] = [];

        foreach ($results as $result) {
            if ($result['status']) {
                $template['languages'][] = [
                    'name'  => $result['name'],
                    'code'  => $result['code'],
                    'image' => $result['image'],
                ];
            }
        }
        if (sizeof($template['languages']) > 1) {
            //selected in selectbox
            $template['language_code'] = $registry->get('language')->getContentLanguageCode();
            foreach ($registry->get('request')->get as $name => $value){
                if ($name == 'content_language_code') {
                    continue;
                }
                $template['hiddens'][$name] = $value;
            }
        } else{
            $template['languages'] = [];
        }
        $view->batchAssign($template);
        return $view->fetch('form/language_flags.tpl');
    }

    /**
     * @param string $html - text that might contain internal links #admin# or #storefront#
     *                          $mode  - 'href' create complete a tag or default just replace URL
     * @param string $type - can be 'message' to convert url into <a> tag or empty
     * @param bool $for_admin - force mode for converting links to admin side from storefront scope (see AIM-class etc)
     *
     * @return string - html code with parsed internal URLs
     * @throws AException
     */
    public function convertLinks($html, $type = '', $for_admin = false){
        $is_admin = (IS_ADMIN === true || $for_admin);
        $route_sections = $is_admin ? ["admin", "storefront"] : ["storefront"];
        foreach ($route_sections as $rt_type) {
            preg_match_all(
                '/(#'.$rt_type.'#rt=){1}[a-z0-9\/_\-\?\&=\%#]{1,255}(\b|\")/',
                $html,
                $matches,
                PREG_OFFSET_CAPTURE
            );
            if ($matches) {
                foreach ($matches[0] as $match) {
                    $href = str_replace('?', '&', $match[0]);

                    if ($rt_type == 'admin') {
                        if ($for_admin && IS_ADMIN !== true) {
                            $href .= '&s='.ADMIN_PATH;
                        }
                        $new_href = str_replace('#admin#', $this->getSecureURL('').'&', $href);
                    } else {
                        $new_href = str_replace('#storefront#', $this->getCatalogURL('').'&', $href);
                    }
                    $new_href = str_replace(['&amp;', '&&', '&?'], '&', $new_href);
                    $new_href = str_replace('?&', '?', $new_href);
                    $new_href = str_replace('&', '&amp;', $new_href);

                    switch($type){
                        case 'message':
                            $new_href = '<a href="' . $new_href . '" target="_blank">#link-text#</a>';
                            break;
                        default:
                            break;
                    }

                    $html = str_replace($match[0], $new_href, $html);
                }
            }
        }

        return $html;
    }

}

/**
 * Class HtmlElementFactory
 */
class HtmlHcaptchaElementFactory extends HtmlElementFactory {

    static private $available_elements = [
        'I' => [
            'type'   => 'input',
            'method' => 'buildInput',
            'class'  => 'InputHtmlElement',
        ],
        'T' => [
            'type'   => 'textarea',
            'method' => 'buildTextarea',
            'class'  => 'TextareaHtmlElement',
        ],
        'S' => [
            'type'   => 'selectbox',
            'method' => 'buildSelectbox',
            'class'  => 'SelectboxHtmlElement',
        ],
        'M' => [
            'type'   => 'multiselectbox',
            'method' => 'buildMultiselectbox',
            'class'  => 'MultiSelectboxHtmlElement',
        ],
        'R' => [
            'type'   => 'radio',
            'method' => 'buildRadio',
            'class'  => 'RadioHtmlElement',
        ],
        'C' => [
            'type'   => 'checkbox',
            'method' => 'buildCheckbox',
            'class'  => 'CheckboxHtmlElement',
        ],
        'G' => [
            'type'   => 'checkboxgroup',
            'method' => 'buildCheckboxgroup',
            'class'  => 'CheckboxgroupHtmlElement',
        ],
        'U' => [
            'type'   => 'file',
            'method' => 'buildFile',
            'class'  => 'FileHtmlElement',
        ],
        'K' => [
            'type'   => 'captcha',
            'method' => 'buildCaptcha',
            'class'  => 'CaptchaHtmlElement',
        ],
        'J' => [
            'type'   => 'recaptcha',
            'method' => 'buildReCaptcha',
            'class'  => 'ReCaptchaHtmlElement',
        ],
        'H' => [
            'type'   => 'hidden',
            'method' => 'buildHidden',
            'class'  => 'HiddenHtmlElement',
        ],
        'P' => [
            'type'   => 'multivalue',
            'method' => 'buildMultivalue',
            'class'  => 'MultivalueHtmlElement',
        ],
        'L' => [
            'type'   => 'multivaluelist',
            'method' => 'buildMultivalueList',
            'class'  => 'MultivalueListHtmlElement',
        ],
        'D' => [
            'type'   => 'date',
            'method' => 'buildDateInput',
            'class'  => 'DateInputHtmlElement',
        ],
        'E' => [
            'type'   => 'email',
            'method' => 'buildEmail',
            'class'  => 'EmailHtmlElement',
        ],
        'N' => [
            'type'   => 'number',
            'method' => 'buildNumber',
            'class'  => 'NumberHtmlElement',
        ],
        'F' => [
            'type'   => 'phone',
            'method' => 'buildPhone',
            'class'  => 'PhoneHtmlElement',
        ],
        'A' => [
            'type'   => 'IPaddress',
            'method' => 'buildIPaddress',
            'class'  => 'IPaddressHtmlElement',
        ],
        'O' => [
            'type'   => 'countries',
            'method' => 'buildCountries',
            'class'  => 'CountriesHtmlElement',
        ],
        'Z' => [
            'type'   => 'zones',
            'method' => 'buildZones',
            'class'  => 'ZonesHtmlElement',
        ],
        'B' => [
            'type'   => 'label',
            'method' => 'buildLabel',
            'class'  => 'LabelHtmlElement',
        ],
        'Q' => [
            'type' => 'hcaptcha',
            'method' => 'buildHcaptcha',
            'class' => 'HcaptchaHtmlElement'
        ],
    ];

    /**
     *  return array of HTML elements supported
     *  array key - code of element
     *  [
     *   type - element type
     *   method - method in html class to get element html
     *   class - element class
     *  ]
     *
     * @static
     * @return array
     */
    static function getAvailableElements()
    {
        return self::$available_elements;
    }

    /**
     * return array of elements indexes for elements which has calculation
     *
     * @static
     * @return array
     */
    static function getElementsCalculation(){
        return self::$elements_with_calculation;
    }

    /**
     * return element type
     *
     * @static
     * @param $code - element code ( from $available_elements )
     * @return null | string
     */
    static function getElementType($code){
        if (!array_key_exists($code, self::$available_elements)){
            return null;
        }
        return self::$available_elements[$code]['type'];
    }

    /**
     * @param $data
     * @return HiddenHtmlElement | MultivalueListHtmlElement | MultivalueHtmlElement | SubmitHtmlElement | InputHtmlElement | PasswordHtmlElement | PaginationHtmlElement | TextAreaHtmlElement | SelectboxHtmlElement | MultiSelectboxHtmlElement | CheckboxHtmlElement | CheckboxGroupHtmlElement | FileHtmlElement | RadioHtmlElement | ButtonHtmlElement | FormHtmlElement | RatingHtmlElement | CaptchaHtmlElement | ReCaptchaHtmlElement | PasswordsetHtmlElement | ResourceHtmlElement | ResourceImageHtmlElement | DateHtmlElement | EmailHtmlElement | NumberHtmlElement | PhoneHtmlElement | IPaddressHtmlElement | CountriesHtmlElement | ZonesHtmlElement | ModalHtmlElement
     * @throws AException
     */
    static function create($data){
        $class = ucfirst($data['type'] . 'HtmlElement');
        if (!class_exists($class)){
            throw new AException(AC_ERR_LOAD, 'Error: Could not load HTML element ' . $data['type'] . '!');
        }
        return new $class($data);
    }
}

/**
 * Class HcaptchaHtmlElement
 * @property string $name
 * @property string $element_id
 * @property string site_key
 * @property string secret_key
 * @property string theme
 * @property string size
 * @property string language_code
 */
class HcaptchaHtmlElement extends HtmlElement{
    /**
     * @return string
     */
    public function getHtml(){

        if (!isset($this->default)) $this->default = '';
        if ($this->value == '' && !empty($this->default)) $this->value = $this->default;
        $this->view->batchAssign(
            [
                'name' => $this->name,
                'id' => $this->element_id,
                'site_key' => $this->site_key,
                'secret_key' => $this->secret_key,
                'theme' => $this->theme,
                'size' => $this->size,
                'language_code' => $this->language_code
            ]
        );

        if (is_object($this->language)
            && sizeof($this->language->getActiveLanguages()) > 1
        ){
            $this->view->assign('multilingual', $this->multilingual);
        }
        if (!empty($this->help_url)){
            $this->view->assign('help_url', $this->help_url);
        }

        return $this->view->fetch('form/hcaptcha.tpl');
    }
}